(*
 * This file is part of Barista.
 * Copyright (C) 2007-2014 Xavier Clerc.
 *
 * Barista is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Barista is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *)

IFNDEF USE_JDK THEN

BARISTA_ERROR =
  | Unable_to_open_writer of (p : Path.t) ->
      Printf.sprintf "unable to open writer (%S)"
        (Path.to_string p)
  | Unable_to_write_data ->
      "unable to write data"
  | Unable_to_close_writer ->
      "unable to close writer"

module Encoding =
  CamomileLibrary.CharEncoding.Configure (CamomileLibraryDefault.Config)

module UTF8Line = CamomileLibrary.ULine.Make (CamomileLibrary.UTF8)

type t = UTF8Line.output_line

external bytes_of_string : string -> Bytes.t =
  "%identity"

external string_of_path : Path.t -> string =
  "%identity"

class output_channel chan = object

  method output str ofs len =
    let bytes = bytes_of_string str in
    OutputStream.write_bytes_from chan bytes ofs len;
    len

  method flush () =
    OutputStream.flush chan

  method close_out () =
    OutputStream.close chan

end

let make_of_stream stream =
  let res =
    stream
    |> new output_channel
    |> new Encoding.uchar_output_channel_of Encoding.utf8 in
  new UTF8Line.output_line res

let make_of_path path =
  (try
    path
    |> string_of_path
    |> open_out
  with _ ->
    fail (Unable_to_open_writer path))
  |> OutputStream.make_of_channel
  |> make_of_stream

external camomile_of_utf8 : UTF8.t -> CamomileLibrary.UTF8.t =
  "%identity"

let write_line writer str =
  try
    str
    |> camomile_of_utf8
    |> writer#put
  with _ ->
    fail Unable_to_write_data

let write_lines writer l =
  try
    List.iter
      (fun str ->
        str
        |> camomile_of_utf8
        |> writer#put)
      l
  with _ ->
    fail Unable_to_write_data

let flush writer =
  try
    writer#flush ()
  with _ ->
    fail Unable_to_write_data

let close writer =
  try
    writer#close_out ()
  with _ ->
    fail Unable_to_close_writer

let close_noerr writer =
  try
    writer#close_out ()
  with _ ->
    ()

ELSE (* USE_JDK *)

BARISTA_ERROR =
  | Unable_to_open_writer of (p : Path.t) ->
      Printf.sprintf "unable to open writer (%S)"
        (Path.to_string p)
  | Unable_to_write_data ->
      "unable to write data"
  | Unable_to_close_writer ->
      "unable to close writer"

type t = java'io'BufferedWriter java_instance

external java_string_of_utf8 : UTF8.t -> java'lang'String java_instance =
  "%identity"

external data_output_stream_of_output_stream : OutputStream.t -> java'io'OutputStream java_instance =
  "%identity"

external file_of_path : Path.t -> java'io'File java_instance =
  "%identity"

let make_of_path path =
  try
    path
    |> file_of_path
    |> Java.make "java.io.FileWriter(java.io.File)"
    |> Java.make "java.io.BufferedWriter(_)"
  with Java_exception _ ->
    fail (Unable_to_open_writer path)

let make_of_stream stream =
  stream
  |> data_output_stream_of_output_stream
  |> Java.make "java.io.OutputStreamWriter(_)"
  |> Java.make "java.io.BufferedWriter(_)"

let write_line writer str =
  try
    str
    |> java_string_of_utf8
    |> Java.call "java.io.BufferedWriter.write(String)" writer;
    Java.call "java.io.BufferedWriter.newLine()" writer
  with Java_exception _ ->
    fail Unable_to_write_data

let write_lines writer l =
  try
    List.iter
      (fun str ->
        str
        |> java_string_of_utf8
        |> Java.call "java.io.BufferedWriter.write(String)" writer;
        Java.call "java.io.BufferedWriter.newLine()" writer)
      l
  with Java_exception _ ->
    fail Unable_to_write_data

let flush writer =
  try
    Java.call "java.io.BufferedWriter.flush()" writer
  with Java_exception _ ->
    fail Unable_to_write_data

let close writer =
  try
    Java.call "java.io.BufferedWriter.close()" writer
  with Java_exception _ ->
    fail Unable_to_close_writer

let close_noerr writer =
  try
    Java.call "java.io.BufferedWriter.close()" writer
  with Java_exception _ ->
    ()

END
