(*
 * This file is part of Barista.
 * Copyright (C) 2007-2014 Xavier Clerc.
 *
 * Barista is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Barista is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *)

(** Parsing of command-line arguments. *)

exception Bad_element of string
(** The exception to be raised when an element associated with a
    command-line switch is invalid. *)

type string_switch = {
    switch_desc : UTF8.t;
      (** The string describing the values accepted by the switch. *)
    handle_string : UTF8.t -> unit;
      (** The function called when the switch is passed. *)
  }
(** The type of switches accepting strings. *)

type choice_switch = {
    switch_options : UTF8.t list;
      (** The list of values accepted by the switch. *)
    handle_choice : UTF8.t -> unit;
      (** The function called when the switch is passed. *)
  }
(** The type of switches accepting strings among a fixed list. *)

type kind =
  | Nothing of (unit -> unit)
      (** Command-line switch accepting no element. *)
  | String of string_switch
      (** Command-line switch accepting a string element. *)
  | Choice of choice_switch
      (** Command-line switch accepting a string among a list. *)
(** The type of argument kinds, that describes the type of elements
    associated with a command-line switch. *)

type switch = {
    identifier : UTF8.t;
      (** Switch identifier, with leading dash. *)
    kind : kind;
      (** Switch kind, containing function to handle switch. *)
    documentation : UTF8.t;
      (** Switch documentation. *)
  }
(** The type of command-line switches. *)

exception Bad_argument of string
(** The exception to be raised when a command-line argument is
    invalid. *)

type others = {
    argument_desc : UTF8.t;
      (** The string describing the values accepted from the command-line. *)
    handle_argument : (UTF8.t -> unit)
      (** The function called to handle command-line arguments. *)
  }
(** The type of functions handling command-line arguments. *)

val parse : switch list -> others option -> UTF8.t array -> int -> unit
(** [parse switches others args idx] parses the command-line represented
    by [args], starting at index [idx]. [switches] contains the switches
    accepted by the program while [others] is used to parse elements that
    are not switches. *)
