(*
 * This file is part of Barista.
 * Copyright (C) 2007-2014 Xavier Clerc.
 *
 * Barista is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Barista is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *)

open Consts

let (++) = UTF8.(++)

type package_names =
  | Full
  | Short
  | None_


(* Functions *)

let print_class_name pn x =
  match pn with
  | Full -> Name.printable_utf8_for_class x
  | Short -> Name.short_utf8_for_class x
  | None_ -> Name.printable_utf8_for_class (snd (Name.split_class_name x))
  

let print_java_type pn x =
  match pn with
  | Full -> Descriptor.external_utf8_of_java_type x
  | Short -> Descriptor.short_utf8_of_java_type x
  | None_ -> Descriptor.shortest_utf8_of_java_type x

let print_java_type_varargs pn x =
  match pn with
  | Full -> Descriptor.external_utf8_of_java_type_varargs x
  | Short -> Descriptor.short_utf8_of_java_type_varargs x
  | None_ -> Descriptor.shortest_utf8_of_java_type_varargs x

let printed_flag = function
  | `Public
  | `Private
  | `Protected
  | `Static
  | `Final
  | `Synchronized
  | `Volatile
  | `Abstract
  | `Strict
  | `Transient
  | `Native
  | `Module -> true
  | `Super
  | `Bridge
  | `Varargs
  | `Interface
  | `Synthetic
  | `Annotation
  | `Enum
  | `Mandated -> false

let rec utf8_of_class_signature (pn : package_names) (varargs : bool) cs =
  let ftp =
    if cs.Signature.formal_type_parameters <> [] then
      let l =
        UTF8.concat_sep_map
          @", "
          (utf8_of_formal_type_parameter pn varargs)
          cs.Signature.formal_type_parameters in
      SPRINTF ("<%s>" l)
    else
      @"" in
  let scs = utf8_of_class_type_signature pn varargs cs.Signature.super_class_signature in
  let sis = List.map (utf8_of_class_type_signature pn varargs) cs.Signature.super_interface_signatures in
  (ftp, scs, sis)
and utf8_of_formal_type_parameter (pn : package_names) (varargs : bool) ftp =
  let fts = utf8_of_field_type_signature pn varargs ftp.Signature.class_bound in
  SPRINTF ("%s%s"
             ftp.Signature.identifier
             (if UTF8.equal fts java_lang_Object then
               (if ftp.Signature.interface_bounds <> [] then
                 let l =
                   UTF8.concat_sep_map
                     @" & "
                     (utf8_of_field_type_signature pn varargs)
                     ftp.Signature.interface_bounds in
                 SPRINTF (" extends %s" l)
               else
                 @"")
             else
               SPRINTF (" extends %s%s"
                          fts
                          (UTF8.concat
                             (List.map
                                (fun x -> @" & " ++ (utf8_of_field_type_signature pn varargs x))
                                ftp.Signature.interface_bounds)))))
and utf8_of_field_type_signature (pn : package_names) (varargs : bool) = function
  | Signature.Class_type_signature cts -> utf8_of_class_type_signature pn varargs cts
  | Signature.Array_type_signature ats -> utf8_of_array_type_signature pn varargs ats
  | Signature.Type_variable_signature tvs -> utf8_of_type_variable_signature pn tvs
and utf8_of_class_type_signature (pn : package_names) (varargs : bool) cts =
  let name = print_class_name pn cts.Signature.qualified_class_name in
  let sign =
    if cts.Signature.type_arguments <> [] then
      let l =
        UTF8.concat_sep_map
          @", "
          (utf8_of_type_argument pn varargs)
          cts.Signature.type_arguments in
      SPRINTF ("<%s>" l)
    else
      @"" in
  SPRINTF ("%s%s" name sign)
and utf8_of_array_type_signature (pn : package_names) (varargs : bool) ats =
  let ts = utf8_of_type_signature pn false ats in
  let va = if varargs then @"..." else @"[]" in
  SPRINTF ("%s%s" ts va)
and utf8_of_type_signature (pn : package_names) (varargs : bool) = function
  | Signature.Field_type_signature fts -> utf8_of_field_type_signature pn varargs fts
  | Signature.Base_type jt -> print_java_type pn jt
and utf8_of_type_variable_signature (_pn : package_names) tvs = tvs
and utf8_of_type_argument (pn : package_names) (varargs : bool) = function
  | Signature.Star -> @"?"
  | Signature.Plus fts ->
      SPRINTF ("%s%s" @"? extends " (utf8_of_field_type_signature pn varargs fts))
  | Signature.Minus fts ->
      SPRINTF ("%s%s" @"? super " (utf8_of_field_type_signature pn varargs fts))
  | Signature.Simple fts ->
      utf8_of_field_type_signature pn varargs fts
(* and utf8_of_class_type_signature_suffix ctss = utf8_of_class_type_signature ctss *)
and utf8_of_throws_signature (pn : package_names) (varargs : bool) = function
  | Signature.Throws_class_type_signature cts ->
      utf8_of_class_type_signature pn varargs cts
  | Signature.Throws_type_variable_signature tvs ->
      utf8_of_type_variable_signature pn tvs

let rec utf8_of_element_value (pn : package_names) = function
  | Annotation.Boolean_value b -> SPRINTF ("%B" b)
  | Annotation.Byte_value i -> SPRINTF ("%d" i)
  | Annotation.Char_value c -> SPRINTF ("%C" c)
  | Annotation.Double_value f -> SPRINTF ("%f" f)
  | Annotation.Float_value f -> SPRINTF ("%f" f)
  | Annotation.Int_value i -> SPRINTF ("%ld" i)
  | Annotation.Long_value i -> SPRINTF ("%Ld" i)
  | Annotation.Short_value i -> SPRINTF ("%d" i)
  | Annotation.String_value s -> SPRINTF ("%S" s)
  | Annotation.Enum_value (c, i) ->
      let c = print_class_name pn c in
      let i = Name.utf8_for_field i in
      SPRINTF ("%s.%s" c i)
  | Annotation.Class_value cn -> print_class_name pn cn
  | Annotation.Annotation_value a -> utf8_of_annotation pn a
  | Annotation.Array_value l  ->
      let l = UTF8.concat_sep @", " (List.map (utf8_of_element_value pn) l) in
      SPRINTF ("[%s]" l)
and utf8_of_pair (pn : package_names) (n, v) =
  SPRINTF ("%s=%s" n (utf8_of_element_value pn v))
and utf8_of_annotation (pn : package_names) (name, values) =
  let name = print_class_name pn name in
  let values =
    if values <> [] then
      let values = UTF8.concat_sep @", " (List.map (utf8_of_pair pn) values) in
      SPRINTF ("(%s)" values)
    else
      @"" in
  SPRINTF ("@%s%s" name values)

let add_annotations buffer (pn : package_names) prefix annotations =
  List.iter
    (fun a ->
      BPRINTF ("%s%s\n" buffer prefix (utf8_of_annotation pn a)))
    annotations

let utf8_of_parents (pn : package_names) is_interface cd =
  let buffer = UTF8Buffer.make () in
  try
    let signature = Attribute.extract_class_signature cd.ClassDefinition.attributes in
    let cs, sc, si = utf8_of_class_signature pn false signature in
    UTF8Buffer.add_string buffer cs;
    if not is_interface then begin
      BPRINTF (" extends %s" buffer sc)
    end;
    if si <> [] then begin
      BPRINTF (" %s %s"
                 buffer
                 (if is_interface then @"extends" else @"implements")
                 (UTF8.concat_sep @", " si))
    end;
    UTF8Buffer.contents buffer
  with Not_found ->
    (match is_interface, cd.ClassDefinition.extends with
    | false, (Some v) ->
        let name = print_class_name pn v in
        BPRINTF (" extends %s" buffer name)
    | _ -> ());
    if cd.ClassDefinition.implements <> [] then begin
      BPRINTF (" %s %s"
                 buffer
                 (if is_interface then @"extends" else @"implements")
                 (UTF8.concat_sep_map
                    @", "
                    (print_class_name pn)
                    cd.ClassDefinition.implements))
    end;
    UTF8Buffer.contents buffer

let tab = @"  "

let add_field buffer (pn : package_names) f =
  add_annotations buffer pn tab (Attribute.extract_annotations (f.Field.attributes :> Attribute.t list));
  let type_ =
    try
      let fts = Attribute.extract_field_signature f.Field.attributes in
      utf8_of_field_type_signature pn false fts
    with Not_found ->
      print_java_type pn (f.Field.descriptor :> Descriptor.java_type) in
  let name = Name.utf8_for_field f.Field.name in
  BPRINTF ("%s%s%s %s;\n"
             buffer
             tab
             (AccessFlag.list_to_utf8 (List.filter printed_flag (f.Field.flags :> AccessFlag.t list)))
          type_
          name)

let add_method buffer (pn : package_names) class_name m =
  let flags, name, desc, attrs =
    match m with
    | Method.Regular mr ->
        mr.Method.flags,
        (Name.utf8_for_method mr.Method.name),
        mr.Method.descriptor,
        mr.Method.attributes
    | Method.Constructor mc ->
        (mc.Method.cstr_flags :> AccessFlag.for_method list),
        class_constructor,
        (mc.Method.cstr_descriptor, `Void),
        mc.Method.cstr_attributes
    | Method.Initializer mi ->
        (mi.Method.init_flags :> AccessFlag.for_method list),
        class_initializer,
        ([], `Void),
        mi.Method.init_attributes in
  let varargs = AccessFlag.mem_method `Varargs flags in
  add_annotations buffer pn tab (Attribute.extract_annotations (attrs :> Attribute.t list));
  UTF8Buffer.add_string buffer tab;
  UTF8Buffer.add_string
    buffer
    (AccessFlag.list_to_utf8 (List.filter printed_flag (flags :> AccessFlag.t list)));
  try
    let fts = Attribute.extract_method_signature attrs in
    if fts.Signature.formal_type_params <> [] then begin
      BPRINTF ("<%s> "
                 buffer
                 (UTF8.concat_sep_map
                    @", "
                    (utf8_of_formal_type_parameter pn false)
                    fts.Signature.formal_type_params))
    end;
    if UTF8.equal name class_initializer then
      UTF8Buffer.add_string buffer @""
    else if UTF8.equal name class_constructor then
      UTF8Buffer.add_string buffer class_name
    else begin
      BPRINTF ("%s %s"
                 buffer
                 (utf8_of_type_signature pn false fts.Signature.return)
                 name)
    end;
    let types =
      if varargs then
        UTF8.concat_sep_map_last
          @", "
          (utf8_of_type_signature pn false)
          (utf8_of_type_signature pn true)
          fts.Signature.types
      else
        UTF8.concat_sep_map
          @", "
          (utf8_of_type_signature pn false)
          fts.Signature.types in
    BPRINTF ("(%s)" buffer types);
    if fts.Signature.throws_signatures <> [] then begin
      BPRINTF (" throws %s"
                 buffer
                 (UTF8.concat_sep_map
                    @", "
                    (utf8_of_throws_signature pn false)
                    fts.Signature.throws_signatures))
    end;
    UTF8Buffer.add_endline buffer @";"
  with Not_found ->
    let params, return = desc in
    if UTF8.equal name class_initializer then
      UTF8Buffer.add_string buffer @""
    else if UTF8.equal name class_constructor then
      UTF8Buffer.add_string buffer class_name
    else begin
      BPRINTF ("%s %s"
                 buffer
                 (print_java_type pn return)
                 name)
    end;
    let types =
      if varargs then
        UTF8.concat_sep_map_last
          @", "
          (print_java_type pn)
          (print_java_type_varargs pn)
          (params :> Descriptor.java_type list)
      else
        UTF8.concat_sep_map
          @", "
          (print_java_type pn)
          (params :> Descriptor.java_type list) in
    BPRINTF ("(%s)" buffer types);
    (try
      let thrown = Attribute.extract_exceptions (attrs :> Attribute.t list) in
      BPRINTF (" throws %s"
                 buffer
                 (UTF8.concat_sep_map
                    @", "
                    (fun x -> print_class_name pn x) thrown))
    with Not_found -> ());
    UTF8Buffer.add_endline buffer @";"

let print_to_buffer buffer (pn : package_names) cp s =
  let cl = ClassLoader.make_of_class_path cp in
  let cd = Lookup.for_class false ~imports:([], [ @"java.lang" ]) cl s in
  let cd = cd.Lookup.value in
  add_annotations buffer pn @"" (Attribute.extract_annotations (cd.ClassDefinition.attributes :> Attribute.t list));
  let flags = (cd.ClassDefinition.access_flags :> AccessFlag.t list) in
  let is_interface = AccessFlag.mem `Interface flags in
  let flags_to_print =
    List.filter
      (fun f ->
        if f = `Abstract then
          not is_interface
        else
          printed_flag f)
      flags in
  let flags = AccessFlag.list_to_utf8 (flags_to_print :> AccessFlag.t list) in
  let name = print_class_name pn cd.ClassDefinition.name in
  BPRINTF ("%s%s %s%s {\n"
             buffer
             flags
             (if is_interface then @"interface" else @"class")
             name
             (utf8_of_parents pn is_interface cd));
  List.iter
    (add_field buffer pn)
    (List.sort Field.compare_according_to_visibility cd.ClassDefinition.fields);
  List.iter
    (add_method buffer pn name)
    (List.sort Method.compare_according_to_visibility cd.ClassDefinition.methods);
  UTF8Buffer.add_string buffer @"}"

let print_to_stream chan (pn : package_names) cp s =
  let buffer = UTF8Buffer.make () in
  print_to_buffer buffer pn cp s;
  let writer = UTF8LineWriter.make_of_stream chan in
  UTF8LineWriter.write_line
    writer
    (UTF8Buffer.contents buffer);
  UTF8LineWriter.flush writer

let print (pn : package_names) cp s =
  print_to_stream OutputStream.stdout pn cp s
