(*
 * This file is part of Barista.
 * Copyright (C) 2007-2014 Xavier Clerc.
 *
 * Barista is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Barista is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *)

IFNDEF USE_JDK THEN

type t = string

let make_of_string str =
  str

let make_of_utf8 str =
  UTF8.to_string str

let basename path =
  path
  |> Filename.basename
  |> UTF8.of_string

let dirname path =
  path
  |> Filename.dirname

let absolute path =
  if Filename.is_relative path then
    Filename.concat (Sys.getcwd ()) path
  else
    path

let is_absolute path =
  not (Filename.is_relative path)

let is_directory path =
  try
    Sys.is_directory path
  with _ ->
    false

let is_file path =
  try
    not (Sys.is_directory path)
  with _ ->
    false

let exists path =
  Sys.file_exists path

let check_suffix suffix path =
  Filename.check_suffix path (UTF8.to_string suffix)

let concat path1 path2 =
  Filename.concat path1 path2

let mkdirs path =
  let rec iter dir =
    match Filename.dirname dir with
    | "." | "/" | "\\" ->
      if not (Sys.file_exists dir) then
        Unix.mkdir dir 0o755
    | parent ->
        iter parent;
	if not (Sys.file_exists dir) then
          Unix.mkdir dir 0o755 in
  try
    if path <> "." then iter path
  with _ ->
    let msg = Printf.sprintf "cannot create directory %S" path in
    raise (Sys_error msg)

let to_string path =
  path

let to_utf8 path =
  path
  |> UTF8.of_string

let equal (path1 : t) (path2 : t) =
  path1 = path2

let compare path1 path2 =
  String.compare path1 path2

let hash path =
  Utils.hash_string path

ELSE (* USE_JDK *)

type t = java'io'File java_instance

external java_string_of_utf8 : UTF8.t -> java'lang'String java_instance =
  "%identity"

external utf8_of_java_string : java'lang'String java_instance -> UTF8.t =
  "%identity"

let make_of_utf8 str =
  str
  |> java_string_of_utf8
  |> Java.make "java.io.File(String)"

let make_of_string str =
  str
  |> JavaString.of_string
  |> Java.make "java.io.File(String)"

let basename path =
  path
  |> Java.call "java.io.File.getName()"
  |> utf8_of_java_string

let dirname path =
  let res =
    path
    |> Java.call "java.io.File.getParentFile()" in
  if Java.is_null res then
    @"."
    |> java_string_of_utf8
    |> Java.make "java.io.File(String)"
  else
    res

let absolute path =
  Java.call "java.io.File.getAbsoluteFile()" path

let is_absolute path =
  Java.call "java.io.File.isAbsolute()" path

let is_directory path =
  Java.call "java.io.File.isDirectory()" path

let is_file path =
  Java.call "java.io.File.isFile()" path

let exists path =
  Java.call "java.io.File.exists()" path

let check_suffix suffix path =
  Java.call "String.endsWith(String)"
    (Java.call "java.io.File.getName()" path)
    (java_string_of_utf8 suffix)

let concat path1 path2 =
  Java.make "java.io.File(java.io.File, String)"
    path1
    (Java.call "java.io.File.getPath()" path2)

let mkdirs path =
  if not (Java.call "java.io.File.mkdirs()" path) then
    let msg = Printf.sprintf "cannot create directory %S"
        (JavaString.to_string @@ Java.call "java.io.File.getPath()" path) in
    raise (Sys_error msg)

let to_string path =
  path
  |> Java.call "java.io.File.getPath()"
  |> JavaString.to_string

let to_utf8 path =
  path
  |> Java.call "java.io.File.getPath()"
  |> utf8_of_java_string

let equal path1 path2 =
  Java.call "java.io.File.equals(_)" path1 path2

let compare path1 path2 =
  Java.call "java.io.File.compareTo(java.io.File)" path1 path2
  |> Int32.to_int

let hash path =
  path
  |> Java.call "java.io.File.hashCode()"
  |> Int32.to_int

END
