(*
 * This file is part of Barista.
 * Copyright (C) 2007-2014 Xavier Clerc.
 *
 * Barista is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Barista is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *)

IFNDEF USE_JDK THEN

type t = Zip.in_file

BARISTA_ERROR =
  | Unable_to_open_archive of (p : Path.t) ->
      Printf.sprintf "unable to open archive (%S)"
        (Path.to_string p)      
  | Unknown_entry of (n : UTF8.t) ->
      Printf.sprintf "unknown archive entry (%S)"
        (UTF8.to_string_noerr n)
  | Unable_to_read_entry of (n : UTF8.t) ->
      Printf.sprintf "unable to read entry (%S)" (UTF8.to_string_noerr n)
  | Unable_to_close_archive ->
      "unable to close archive"

external bytes_of_string : string -> Bytes.t =
  "%identity"

external archive_entry_of_zip_entry : Zip.entry -> ArchiveEntry.t =
  "%identity"

external zip_entry_of_archive_entry : ArchiveEntry.t -> Zip.entry =
  "%identity"

let make_of_path path =
  try
    path
    |> Path.to_string
    |> Zip.open_in
  with _ ->
    fail (Unable_to_open_archive path)

let get_comment archive =
  archive
  |> Zip.comment
  |> UTF8.of_string

let find_entry archive name =
  try
    name
    |> UTF8.to_string
    |> Zip.find_entry archive
    |> archive_entry_of_zip_entry
  with _ ->
    fail (Unknown_entry name)

let iter_entries f archive =
  archive
  |> Zip.entries
  |> List.iter (fun entry -> f @@ archive_entry_of_zip_entry entry)

let fold_entries f zero archive =
  archive
  |> Zip.entries
  |> List.fold_left (fun acc entry -> f acc @@ archive_entry_of_zip_entry entry) zero

let stream_of_entry archive entry =
  try
    entry
    |> zip_entry_of_archive_entry
    |> Zip.read_entry archive
    |> bytes_of_string
    |> InputStream.make_of_bytes
  with _ ->
    fail (Unable_to_read_entry (ArchiveEntry.get_filename entry))

let bytes_of_entry archive entry =
  try
    entry
    |> zip_entry_of_archive_entry
    |> Zip.read_entry archive
    |> bytes_of_string
  with _ ->
    fail (Unable_to_read_entry (ArchiveEntry.get_filename entry))

let close archive =
  try
    Zip.close_in archive
  with _ ->
    fail Unable_to_close_archive

let close_noerr archive =
  try
    Zip.close_in archive
  with _ ->
    ()

ELSE (* USE_JDK *)

type t = java'util'zip'ZipFile java_instance

BARISTA_ERROR =
  | Unable_to_open_archive of (p : Path.t) ->
      Printf.sprintf "unable to open archive (%S)"
        (Path.to_string p)      
  | Unknown_entry of (n : UTF8.t) ->
      Printf.sprintf "unknown archive entry (%S)"
        (UTF8.to_string_noerr n)
  | Unable_to_read_entry of (n : UTF8.t) ->
      Printf.sprintf "unable to read entry (%S)" (UTF8.to_string_noerr n)
  | Unable_to_close_archive ->
      "unable to close archive"

external utf8_of_java_string : java'lang'String java_instance -> UTF8.t =
  "%identity"

external java_string_of_utf8 : UTF8.t -> java'lang'String java_instance =
  "%identity"

external bytes_of_byte_array : int JavaByteArray.t -> Bytes.t =
  "%identity"

external input_stream_of_data_input_stream : java'io'DataInputStream java_instance -> InputStream.t =
  "%identity"

external file_of_path : Path.t -> java'io'File java_instance =
  "%identity"

external archive_entry_of_zip_entry : java'util'zip'ZipEntry java_instance -> ArchiveEntry.t =
  "%identity"

external zip_entry_of_archive_entry : ArchiveEntry.t -> java'util'zip'ZipEntry java_instance =
  "%identity"

let make_of_path path =
  try
    path
    |> file_of_path
    |> Java.make "java.util.zip.ZipFile(java.io.File)"
  with Java_exception _ ->
    fail (Unable_to_open_archive path)

let get_comment archive =
  let res =
    archive
    |> Java.call "java.util.zip.ZipFile.getComment()" in
  (if Java.is_null res then
    Java.make "String()" ()
  else
    res)
  |> utf8_of_java_string

let find_entry archive name =
  let res =
    name
    |> java_string_of_utf8
    |> Java.call "java.util.zip.ZipFile.getEntry(_)" archive in
  if Java.is_null res then
    fail (Unknown_entry name)
  else
    archive_entry_of_zip_entry res

let iter_entries f archive =
  let enum = Java.call "java.util.zip.ZipFile.entries()" archive in
  while Java.call "java.util.Enumeration.hasMoreElements()" enum do
    enum
    |> Java.call "java.util.Enumeration.nextElement()"
    |> Java.cast "java.util.zip.ZipEntry"
    |> archive_entry_of_zip_entry
    |> f
  done

let fold_entries f zero archive =
  let acc = ref zero in
  let enum = Java.call "java.util.zip.ZipFile.entries()" archive in
  while Java.call "java.util.Enumeration.hasMoreElements()" enum do
    let elem =
      enum
      |> Java.call "java.util.Enumeration.nextElement()"
      |> Java.cast "java.util.zip.ZipEntry"
      |> archive_entry_of_zip_entry in
    acc := f !acc elem
  done;
  !acc

let stream_of_entry archive entry =
  try
    entry
    |> zip_entry_of_archive_entry
    |> Java.call "java.util.zip.ZipFile.getInputStream(_)" archive
    |> Java.make "java.io.DataInputStream(_)"
    |> input_stream_of_data_input_stream
  with _ ->
    let filename =
      entry
      |> zip_entry_of_archive_entry
      |> Java.call "java.util.zip.ZipEntry.getName()"
      |> utf8_of_java_string in
    fail (Unable_to_read_entry filename)

let bytes_of_entry archive entry =
  let buffer_size = 65536l in
  try
    let buffer = Java.make_array "byte[]" 65536l in
    let data = Java.make "java.io.ByteArrayOutputStream(_)" buffer_size in
    let is =
      entry
      |> zip_entry_of_archive_entry
      |> Java.call "java.util.zip.ZipFile.getInputStream(_)" archive in
    let last_read = ref buffer_size in
    while !last_read = buffer_size do
      last_read := Java.call "java.io.InputStream.read(_)" is buffer;
      if !last_read <> -1l then
        Java.call "java.io.ByteArrayOutputStream.write(_,_,_)" data buffer 0l !last_read;
    done;
    data
    |> Java.call "java.io.ByteArrayOutputStream.toByteArray()"
    |> bytes_of_byte_array
  with _ ->
    let filename =
      entry
      |> zip_entry_of_archive_entry
      |> Java.call "java.util.zip.ZipEntry.getName()"
      |> utf8_of_java_string in
    fail (Unable_to_read_entry filename)

let close archive =
  try
    archive
    |> Java.call "java.util.zip.ZipFile.close()"
  with Java_exception _ ->
    fail Unable_to_close_archive

let close_noerr archive =
  try
    archive
    |> Java.call "java.util.zip.ZipFile.close()"
  with Java_exception _ ->
    ()

END
