(*
 * This file is part of Barista.
 * Copyright (C) 2007-2014 Xavier Clerc.
 *
 * Barista is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Barista is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *)

type result =
  | Class_definition of ClassDefinition.t
  | Exception of exn

let get_parent loader name =
  try
    Class_definition (ClassLoader.find_class_name loader name)
  with e ->
    Exception e

let map_definitions_to_names l =
  List.map (fun cd -> cd.ClassDefinition.name) l

let parent_class_definitions fail loader cd =
  let rec parents acc curr =
    match cd.ClassDefinition.extends with
    | Some parent_name ->
        (match get_parent loader parent_name with
        | Class_definition parent ->
            parents (curr :: acc) parent
        | Exception ex ->
            if fail then raise ex else List.rev (curr :: acc))
    | None -> List.rev (curr :: acc) in
  parents [] cd

let parent_class_names fail loader cd =
  parent_class_definitions fail loader cd
  |> map_definitions_to_names

module ClassNameSet = Set.Make (struct
  type t = Name.for_class
  let compare cn1 cn2 =
    Name.compare_for_class cn1 cn2
end)

let all_parent_class_definitions fail loader cd =
  let rec all_parents loader acc seen curr =
    let curr_name = curr.ClassDefinition.name in
    let acc, seen =
      if ClassNameSet.mem curr_name seen then
        acc, seen
      else
        curr :: acc, ClassNameSet.add curr_name seen in
    let parent_names =
      match curr.ClassDefinition.extends with
      | Some ext -> ext :: curr.ClassDefinition.implements
      | None -> curr.ClassDefinition.implements in
    List.fold_left
      (fun (acc, seen) parent_name ->
        match get_parent loader parent_name with
        | Class_definition parent ->
            all_parents loader acc seen parent
        | Exception ex ->
            if fail then raise ex else (acc, seen))
      (acc, seen)
      parent_names in
  all_parents loader [] ClassNameSet.empty cd
  |> fst
  |> List.rev

let all_parent_class_names fail loader cd =
  all_parent_class_definitions fail loader cd
  |> map_definitions_to_names
