(*
 * This file is part of Barista.
 * Copyright (C) 2007-2014 Xavier Clerc.
 *
 * Barista is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Barista is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *)

type pcdata = UTF8.t

type tag = {
    tag : UTF8.t;
    attrs : (UTF8.t * UTF8.t) list;
    children : t list;
  }

and t =
  | Pcdata of pcdata
  | Tag of tag

let pcdata str =
  Pcdata str

let tag ~tag ~attrs ~children =
  Tag { tag; attrs; children }

let escape s =
  let len = UTF8.length s in
  let buff = UTF8Buffer.make_of_size (2 * len) in
  for i = 0 to pred len do
    match UTF8.get s i with
    | @'<' ->
        UTF8Buffer.add_string buff @"&lt;"
    | @'>' ->
        UTF8Buffer.add_string buff @"&gt;"
    | @'\'' ->
        UTF8Buffer.add_string buff @"&apos;"
    | @'\"' ->
        UTF8Buffer.add_string buff @"&quot;"
    | @'&' ->
        UTF8Buffer.add_string buff @"&amp;"
    | @'\t' ->
        UTF8Buffer.add_string buff @"&#9;"
    | @'\n' ->
        UTF8Buffer.add_string buff @"&#10;"
    | ch ->
        UTF8Buffer.add_char buff ch
  done;
  UTF8Buffer.contents buff

let rec dump level xml buff =
  let add_string str = UTF8Buffer.add_string buff str in
  let add_endline str = UTF8Buffer.add_endline buff str in
  let add_indent () =
    add_endline @"";
    for _i = 1 to level do
      add_string @"  "
    done in
  match xml with
  | Pcdata pcdata ->
      add_string pcdata
  | Tag { tag; attrs; children } ->
      add_indent ();
      BPRINTF ("<%s" buff tag);
      List.iter
        (fun (name, value) ->
          BPRINTF (" %s=\"%s\"" buff name (escape value)))
        attrs;
      if children = [] then
        add_string @"/>"
      else begin
        add_string @">";
        let needs_ident =
          List.fold_left
            (fun acc child ->
              dump (succ level) child buff;
              acc || (match child with Tag _ -> true | _ -> false))
            false
            children in
        if needs_ident then
          add_indent ();
        BPRINTF ("</%s>" buff tag)
      end

let dump xml buff =
  UTF8Buffer.add_string buff @"<?xml version=\"1.0\" encoding=\"UTF-8\"?>";
  dump 0 xml buff;
  UTF8Buffer.add_endline buff @""

let to_utf8 xml =
  let buff = UTF8Buffer.make_of_size 2048 in
  dump xml buff;
  UTF8Buffer.contents buff
