(*
 * This file is part of Barista.
 * Copyright (C) 2007-2014 Xavier Clerc.
 *
 * Barista is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Barista is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *)


type 'a t = {
    mutable current : 'a array;
    mutable next : int;
  }

type index = Utils.u2

let default_capacity = 128

let make len cap init =
  if (cap < 0) || (len < 0) then
    invalid_arg "BaristaLibrary.ExtendableArray.make";
  let cap = Utils.max_int cap len in
  if cap <= (Utils.max_u2_value + 1) then
    { current = Array.make cap init;
      next = len; }
  else
    invalid_arg "BaristaLibrary.ExtendableArray.make"

let from_array exn arr x =
  let len = Array.length arr in
  let size = ref default_capacity in
  while (!size <= Utils.max_u2_value) && (!size < len) do
    size := 2 * !size
  done;
  let size = Utils.min_int Utils.max_u2_value !size in
  if (size >= len) then
    { current =
      Array.init
        size
        (fun idx -> if idx < len then arr.(idx) else x);
      next = len; }
  else
    raise exn

let to_array arr =
  Array.sub arr.current 0 arr.next

let length arr =
  arr.next

let capacity arr =
  Array.length arr.current

let get arr idx =
  let idx = (idx : Utils.u2 :> int) in
  if (idx < 0) || (idx >= arr.next) then
    invalid_arg "BaristaLibrary.ExtendableArray.get"
  else
    arr.current.(idx)

let set arr idx x =
  let idx = (idx : Utils.u2 :> int) in
  if (idx < 0) || (idx >= arr.next) then
    invalid_arg "BaristaLibrary.ExtendableArray.set"
  else
    arr.current.(idx) <- x

external unsafe_u2 : int -> Utils.u2 =
  "%identity"

let find p arr =
  let len = arr.next in
  let idx = ref 0 in
  while (!idx < len) && not (p arr.current.(!idx)) do
    incr idx
  done;
  if (!idx < len) then
    unsafe_u2 !idx
  else
    raise Not_found

let rec add exn arr x z addit =
  let len = Array.length arr.current in
  let size = if addit then 2 else 1 in
  let next = arr.next in
  if next + size <= len then begin
    arr.current.(arr.next) <- x;
    if addit then arr.current.(succ arr.next) <- z;
    arr.next <- next + size;
    unsafe_u2 next
  end else begin
    if len >= Utils.max_u2_value then raise exn;
    let new_array =
      Array.init
        (Utils.min_int Utils.max_u2_value (2 * len))
        (fun idx -> if idx < len then arr.current.(idx) else z) in
    arr.current <- new_array;
    add exn arr x z addit
  end

let add_if_not_found exn p arr x z addit =
  try
    find p arr
  with Not_found ->
    add exn arr x z addit

let equal eq arr1 arr2 =
  (* do not rely on Utils.array_equal because we are only interested
     in elements before x.next *)
  if arr1.next = arr2.next then
    let idx = ref 0 in
    while (!idx < arr1.next) && (eq arr1.current.(!idx) arr2.current.(!idx)) do
      incr idx
    done;
    !idx = arr1.next
  else
    false

let compare cmp arr1 arr2 =
  (* do not rely on Utils.array_compare because we are only interested
     in elements before arr1.next *)
  if arr1.next = arr2.next then
    let rec iter idx =
      if idx < arr1.next then
        let res = cmp arr1.current.(idx) arr2.current.(idx) in
        if res = 0 then iter (succ idx) else res
      else
        0 in
    iter 0
  else
    Pervasives.compare arr1.next arr2.next

let hash h arr =
  (* do not rely on Utils.array_hash because we are only interested
     in elements before x.next *)
  let res = ref 0 in
  for idx = 0 to Utils.min_int (pred arr.next) 31 do
    res := !res + (h arr.current.(idx))
  done;
  !res

let to_string f arr =
  (* do not rely on Utils.string_of_array because we are only interested
     in elements before arr.next *)
  if arr.next = 0 then begin
    String.copy "[||]"
  end else begin
    let buff = Buffer.create 128 in
    Buffer.add_string buff "[|";
    Buffer.add_string buff (f arr.current.(0));
    for idx = 1 to pred arr.next do
      Buffer.add_string buff "; ";
      Buffer.add_string buff (f arr.current.(idx))
    done;
    Buffer.add_string buff "|]";
    Buffer.contents buff
  end
