(*
 * This file is part of Barista.
 * Copyright (C) 2007-2014 Xavier Clerc.
 *
 * Barista is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Barista is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *)

(** References to Java elements as used by method handles. *)


(** {6 Types} *)

type for_field = Name.for_class * Name.for_field * Descriptor.for_field
(** Type for references to a field (class name, field name, and field descriptor). *)

type for_method = Name.for_class * Name.for_method * Descriptor.for_method
(** Type for references to a method (class name, method name, and method descriptor). *)

type for_constructor = Name.for_class * (Descriptor.for_parameter list)
(** Type for references to a constructor (class name, and parameter list). *)

type t =
  | GetField of for_field
    (** Reference equivalent to a 'getField' call. *)
  | GetStatic of for_field
    (** Reference equivalent to a 'getStatic' call. *)
  | PutField of for_field
     (** Reference equivalent to a 'putField' call. *)
  | PutStatic of for_field
    (** Reference equivalent to a 'putStatic' call. *)
  | InvokeVirtual of for_method
    (** Reference equivalent to a 'invokeVirtual' call. *)
  | InvokeStatic of for_method
    (** Reference equivalent to a 'invokeStatic' call. *)
  | InvokeSpecial of for_method
    (** Reference equivalent to a 'invokeSpecial' call. *)
  | NewInvokeSpecial of for_constructor
    (** Reference equivalent to a 'newInvokeSpecial' call. *)
  | InvokeInterface of for_method
    (** Reference equivalent to a 'invokeInterface' call. *)
(** Type for references to an element from a method handle. *)

type kind =
  | REF_getField (** Reference equivalent to a 'getField' call. *)
  | REF_getStatic (** Reference equivalent to a 'getStatic' call. *)
  | REF_putField (** Reference equivalent to a 'putField' call. *)
  | REF_putStatic (** Reference equivalent to a 'putStatic' call. *)
  | REF_invokeVirtual (** Reference equivalent to a 'invokeVirtual' call. *)
  | REF_invokeStatic (** Reference equivalent to a 'invokeStatic' call. *)
  | REF_invokeSpecial (** Reference equivalent to a 'invokeSpecial' call. *)
  | REF_newInvokeSpecial (** Reference equivalent to a 'newInvokeSpecial' call. *)
  | REF_invokeInterface (** Reference equivalent to a 'invokeInterface' call. *)
(** Type for reference kinds to an element from a method handle. *)


(** {6 Exception} *)

BARISTA_ERROR =
  | Invalid_kind of Utils.u1


(** {6 Functions} *)

val u1_of_kind : kind -> Utils.u1
(** Converts the passed kind into its corresponding code, as used in the
    classfile format. *)

val kind_of_u1 : Utils.u1 -> kind
(** Converts the passed integer into its corresponding kind, as used in the
    classfile format.

    Raises [Exception] if the passed integer is not a valid kind. *)

val utf8_for_field : for_field -> UTF8.t
(** Converts the passed field into a string. *)

val utf8_for_method : for_method -> UTF8.t
(** Converts the passed method into a string. *)

val utf8_for_constructor : for_constructor -> UTF8.t
(** Converts the passed constructor into a string. *)

val to_utf8 : t -> UTF8.t
(** Converts the passed reference into a string. *)

val equal_for_field : for_field -> for_field -> bool
(** Equality over fields. *)

val equal_for_method : for_method -> for_method -> bool
(** Equality over methods. *)

val equal_for_constructor : for_constructor -> for_constructor -> bool
(** Equality over constructors. *)

val equal : t -> t -> bool
(** Equality over references. *)

val compare_for_field : for_field -> for_field -> int
(** Comparison over field. *)

val compare_for_method : for_method -> for_method -> int
(** Comparison over methods. *)

val compare_for_constructor : for_constructor -> for_constructor -> int
(** Comparison over constructors. *)

val compare : t -> t -> int
(** Comparison over references. *)

val hash_for_field : for_field -> int
(** Hash function over fields. *)

val hash_for_method : for_method -> int
(** Hash function over methods. *)

val hash_for_constructor : for_constructor -> int
(** Hash function over constructors. *)

val hash : t -> int
(** Hash function over references. *)
