(*
 * This file is part of Barista.
 * Copyright (C) 2007-2014 Xavier Clerc.
 *
 * Barista is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Barista is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *)


(* Base types and function *)

type method_handle =
  [ `getField of Reference.for_field
  | `getStatic of Reference.for_field
  | `putField of Reference.for_field
  | `putStatic of Reference.for_field
  | `invokeVirtual of Reference.for_method
  | `invokeStatic of Reference.for_method
  | `invokeSpecial of Reference.for_method
  | `newInvokeSpecial of Reference.for_constructor
  | `invokeInterface of Reference.for_method ]

let equal_method_handle x y =
  match x, y with
  | `getField fr1, `getField fr2
  | `getStatic fr1, `getStatic fr2
  | `putField fr1, `putField fr2
  | `putStatic fr1, `putStatic fr2 -> Reference.equal_for_field fr1 fr2
  | `invokeVirtual mr1, `invokeVirtual mr2
  | `invokeStatic mr1, `invokeStatic mr2
  | `invokeSpecial mr1, `invokeSpecial mr2
  | `invokeInterface mr1, `invokeInterface mr2 -> Reference.equal_for_method mr1 mr2
  | `newInvokeSpecial cr1, `newInvokeSpecial cr2 -> Reference.equal_for_constructor cr1 cr2
  | _ -> false

let compare_method_handle x y =
  match x, y with
  | `getField fr1, `getField fr2
  | `getStatic fr1, `getStatic fr2
  | `putField fr1, `putField fr2
  | `putStatic fr1, `putStatic fr2 -> Reference.compare_for_field fr1 fr2
  | `invokeVirtual mr1, `invokeVirtual mr2
  | `invokeStatic mr1, `invokeStatic mr2
  | `invokeSpecial mr1, `invokeSpecial mr2
  | `invokeInterface mr1, `invokeInterface mr2 -> Reference.compare_for_method mr1 mr2
  | `newInvokeSpecial cr1, `newInvokeSpecial cr2 -> Reference.compare_for_constructor cr1 cr2
  | _ -> Pervasives.compare x y

let hash_method_handle = function
  | `getField f -> 1 + (Reference.hash_for_field f)
  | `getStatic f -> 2 + (Reference.hash_for_field f)
  | `putField f -> 3 + (Reference.hash_for_field f)
  | `putStatic f -> 4 + (Reference.hash_for_field f)
  | `invokeVirtual m -> 5 + (Reference.hash_for_method m)
  | `invokeStatic m -> 6 + (Reference.hash_for_method m)
  | `invokeSpecial m -> 7 + (Reference.hash_for_method m)
  | `newInvokeSpecial c -> 8 + (Reference.hash_for_constructor c)
  | `invokeInterface m -> 9 + (Reference.hash_for_method m)

type method_argument =
  [ `String of UTF8.t
  | `Class of Name.for_class
  | `Integer of int32
  | `Long of int64
  | `Float of float
  | `Double of float
  | `MethodHandle of method_handle
  | `MethodType of Descriptor.for_method ]

let equal_method_argument x y =
  match x, y with
  | `String s1, `String s2 -> UTF8.equal s1 s2
  | `Class cn1, `Class cn2 -> Name.equal_for_class cn1 cn2
  | `Integer i1, `Integer i2 -> i1 = i2
  | `Long l1, `Long l2 -> l1 = l2
  | `Float f1, `Float f2 -> f1 = f2
  | `Double d1, `Double d2 -> d1 = d2
  | `MethodHandle mh1, `MethodHandle mh2 -> equal_method_handle mh1 mh2
  | `MethodType mt1, `MethodType mt2 -> Descriptor.equal_for_method mt1 mt2
  | _ -> false

let compare_method_argument x y =
  match x, y with
  | `String s1, `String s2 -> UTF8.compare s1 s2
  | `Class cn1, `Class cn2 -> Name.compare_for_class cn1 cn2
  | `Integer i1, `Integer i2 -> Pervasives.compare i1 i2
  | `Long l1, `Long l2 -> Pervasives.compare l1 l2
  | `Float f1, `Float f2 -> Pervasives.compare f1 f2
  | `Double d1, `Double d2 -> Pervasives.compare d1 d2
  | `MethodHandle mh1, `MethodHandle mh2 -> compare_method_handle mh1 mh2
  | `MethodType mt1, `MethodType mt2 -> Descriptor.compare_for_method mt1 mt2
  | _ -> Pervasives.compare x y

let hash_method_argument = function
  | `String s -> 1 + (UTF8.hash s)
  | `Class cn -> 2 + (Name.hash_for_class cn)
  | `Integer i -> 3 + (Utils.universal_hash i)
  | `Long l -> 4 + (Utils.universal_hash l)
  | `Float f -> 5 + (Utils.universal_hash f)
  | `Double d -> 6 + (Utils.universal_hash d)
  | `MethodHandle mh -> 7 + (hash_method_handle mh)
  | `MethodType mt -> 8 + (Descriptor.hash_for_method mt)

type method_specifier = method_handle * (method_argument list)

let equal_method_specifier (mh1, mal1) (mh2, mal2) =
  (equal_method_handle mh1 mh2)
    && (Utils.list_equal equal_method_argument mal1 mal2)

let compare_method_specifier (mh1, mal1) (mh2, mal2) =
  let res = compare_method_handle mh1 mh2 in
  if res = 0 then
    Utils.list_compare compare_method_argument mal1 mal2
  else
    res

let hash_method_specifier (mh, mal) =
  (hash_method_handle mh) + (Utils.list_hash hash_method_argument mal)


(* Structure used for encoding *)

type methods = method_specifier ExtendableArray.t

BARISTA_ERROR =
  | Too_large of (x : int) ->
      Printf.sprintf "bootstrap array is too large (%d)" x

let _ = fail (* to avoid warning "unused ``fail''" *)

let dummy_element =
  `getField (Name.make_for_class_from_external @"dummy_package.DummyClass",
             Name.make_for_field @"dummyField",
             `Boolean),
  []

let make_methods () =
  ExtendableArray.make 0 128 dummy_element

let is_empty m =
  (ExtendableArray.length m) = 0

let add_method_specifier m ms =
  ExtendableArray.add_if_not_found
    (Exception (Too_large (ExtendableArray.length m)))
    (fun x ->
      (x != dummy_element) && (equal_method_specifier ms x))
    m
    ms
    dummy_element
    false

let add m ms =
  let _ =
    ExtendableArray.add
      (Exception (Too_large (ExtendableArray.length m)))
      m
      ms
      dummy_element
      false in
  ()
