(*
 * This file is part of Barista.
 * Copyright (C) 2007-2014 Xavier Clerc.
 *
 * Barista is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Barista is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *)


type container =
  | Stack
  | Queue
  | Modifiable_stack

exception Already_present

(* returns add, add', is_empty, and take operations *)
let make_pending = function
  | Stack ->
      let cont = Stack.create () in
      let add x =
        try
          Stack.iter
            (fun y ->
              if ControlFlow.equal_vertex x y then
                raise Already_present)
            cont;
          Stack.push x cont
        with Already_present -> () in
      let is_empty () = Stack.is_empty cont in
      let take () = Stack.pop cont in
      add, add, is_empty, take
  | Queue ->
      let cont = Queue.create () in
      let add x =
        try
          Queue.iter
            (fun y ->
              if ControlFlow.equal_vertex x y then
                raise Already_present)
            cont;
          Queue.add x cont
        with Already_present -> () in
      let is_empty () = Queue.is_empty cont in
      let take () = Queue.take cont in
      add, add, is_empty, take
  | Modifiable_stack ->
      let cont = ref [] in (* head is stack top *)
      let add_top x =
        let l = List.filter (fun y -> not (ControlFlow.equal_vertex x y)) !cont in
        cont := x :: l in
      let add_bottom x =
        if not (List.exists (fun y -> ControlFlow.equal_vertex x y) !cont) then
          cont := !cont @ [ x ] in
      let is_empty () = !cont = [] in
      let take () =
        match !cont with
        | hd :: tl -> cont := tl; hd
        | [] -> raise Stack.Empty in
      add_top, add_bottom, is_empty, take

let map_graph f g =
  ControlFlow.map_graph
    (fun x y -> f x, y)
    (fun x _ -> x)
    (fun x _ _ -> x)
    g

let visit_graph ~container ~init_mark ~init_root ~visit_node g =
  let pending_add, pending_add', pending_is_empty, pending_take = make_pending container in
  (* create a new graph with empty marks *)
  let g = map_graph init_mark g in
  (* mark the root *)
  let root = ControlFlow.root_of_graph g in
  init_root root;
  pending_add root;
  (* visit the graph *)
  while not (pending_is_empty ()) do
    let vertex = pending_take () in
    let edge =
      try
        let edge = ControlFlow.edge_of_vertex g vertex in
        Some (snd edge)
      with Not_found -> None in
    let handlers = ControlFlow.handlers_of_vertex g vertex in
    visit_node pending_add pending_add' vertex edge handlers g
  done;
  g

let remove_dead_code g =
  let g =
    visit_graph
      ~container:Stack
      ~init_mark:(fun x -> ref false, x)
      ~init_root:(fun v ->
        let lbl, _ = ControlFlow.label_of_vertex v in
        lbl := true)
      ~visit_node:(fun pending_add _ _ edge handlers _ ->
        let mark v =
          let lbl, _ = ControlFlow.label_of_vertex v in
          if not !lbl then begin
            lbl := true;
            pending_add v
          end in
        (match edge with
        | Some (ControlFlow.Unconditional_edge (_, dst)) ->
            mark dst
        | Some (ControlFlow.Conditional_edge (_, ifso, ifno)) ->
            mark ifso;
            mark ifno
        | Some (ControlFlow.Switch_edge (_, dest, dests)) ->
            mark dest;
            List.iter mark dests
        | None -> ());
        List.iter (fun (_, _, dst) -> mark dst) handlers)
      g in
  (* remove unvisited nodes *)
  let unvisited =
    ControlFlow.fold_vertices
      (fun v acc ->
        let lbl, _ = ControlFlow.label_of_vertex v in
        if not !lbl then v :: acc else acc)
      g
      [] in
  let g = ControlFlow.remove_vertices g unvisited in
  map_graph snd g

let optimize_jumps g =
  let rec follow_link seen v =
    let instrs = ControlFlow.instructions_of_vertex v in
    if instrs = [] then
      match ControlFlow.edge_of_vertex g v with
      | _, ControlFlow.Unconditional_edge (_, v') ->
          if Utils.list_mem (==) v' seen then
            v
          else
            follow_link (v' :: seen) v'
      | _ -> v
    else
      v in
  let rec follow_jump seen seq v =
    let instrs = ControlFlow.instructions_of_vertex v in
    if instrs = [] then
      match snd (ControlFlow.edge_of_vertex g v) with
      | ControlFlow.Unconditional_edge (seq', dst) ->
          if Utils.list_mem (==) dst seen then
            ControlFlow.Unconditional_edge (seq, v)
          else
            follow_jump (dst :: seen) (seq && seq') dst
      | ControlFlow.Conditional_edge (jk, dst1, dst2) ->
          ControlFlow.Conditional_edge (jk,
                                        follow_link (dst1 :: seen) dst1,
                                        follow_link (dst2 :: seen) dst2)
      | ControlFlow.Switch_edge (sk, dst, l) ->
          ControlFlow.Switch_edge (sk,
                                   follow_link (dst :: seen) dst,
                                   List.map (fun x -> follow_link (x :: seen) x) l)
    else
      ControlFlow.Unconditional_edge (seq, v) in
  let eq_edge = ControlFlow.equal_edge ControlFlow.equal_vertex in
  let changes_edges, changes_handlers =
    ControlFlow.fold_vertices
      (fun v (acc_edges, acc_handlers) ->
        let acc_edges =
          try
            let lbl, e = ControlFlow.edge_of_vertex g v in
            let e' = match e with
            | ControlFlow.Unconditional_edge (seq, dst) ->
                follow_jump [v] seq dst
            | ControlFlow.Conditional_edge (jk, dst1, dst2) ->
                ControlFlow.Conditional_edge (jk, follow_link [v] dst1, follow_link [v] dst2)
            | ControlFlow.Switch_edge (sk, dst, l) ->
                ControlFlow.Switch_edge (sk, follow_link [v] dst, List.map (follow_link [v]) l) in
            if eq_edge e e' then acc_edges else (v, e', lbl) :: acc_edges
          with Not_found -> acc_edges in
        let handlers = ControlFlow.handlers_of_vertex g v in
        let handlers =
          List.fold_left
            (fun acc (lbl, cn, dst) ->
              let dst' = follow_link [v] dst in
              if ControlFlow.equal_vertex dst dst' then
                acc
              else
                (lbl, cn, dst') :: acc)
            []
            handlers in
        let acc_handlers =
          if handlers = [] then
            acc_handlers
          else
            (v, List.rev handlers) :: acc_handlers in
        (acc_edges, acc_handlers))
      g
      ([], []) in
  let g =
    List.fold_left
      (fun acc (v, e, lbl) -> ControlFlow.add_edge acc v e lbl)
      g
      changes_edges in
  List.fold_left
    (fun acc (v, l) ->
      ControlFlow.set_handlers acc v l)
    g
    (List.rev changes_handlers)

let optimize_switches g =
  let changes_edges =
    ControlFlow.fold_vertices
      (fun v acc ->
        try
          let lbl, e = ControlFlow.edge_of_vertex g v in
          match e with
          | ControlFlow.Switch_edge (ControlFlow.Lookup vals, def, dests) ->
              if (vals <> []) && ((List.length vals) = (List.length dests)) then begin
                let min_val, max_val =
                  List.fold_left
                    (fun (min_v, max_v) elem ->
                      let min_v = Utils.min_s4 min_v elem in
                      let max_v = Utils.max_s4 max_v elem in
                      (min_v, max_v))
                    (Utils.max_s4_value, Utils.min_s4_value)
                    vals in
                let min_val = (min_val :> int32) in
                let max_val = (max_val :> int32) in
                let nb_vals =
                  Int64.succ
                    (Int64.sub
                       (Int64.of_int32 max_val)
                       (Int64.of_int32 min_val)) in
                let size_table = Int64.succ nb_vals in
                let size_lookup = Int64.mul 2L (Int64.of_int (List.length vals)) in
                if size_table < size_lookup then begin
                  let original =
                    List.map2
                      (fun x y -> (x : Utils.s4 :> int32), y)
                      vals
                      dests in
                  let rec for32_down (idx : int32) (min_idx : int32) acc =
                    let dest =
                      try
                        Utils.list_assoc
                          (fun x y -> Int32.compare x y = 0)
                          idx original
                      with Not_found -> def in
                    let acc = dest :: acc in
                    if idx = min_idx then
                      acc
                    else for32_down (Int32.pred idx) min_idx acc in
                  let l = for32_down max_val min_val [] in
                  (v, lbl, min_val, max_val, def, l) :: acc
                end else
                  acc
              end else
                acc
          | _ -> acc
        with Not_found -> acc)
      g
      [] in
  List.fold_left
    (fun acc (v, lbl, min_val, max_val, def, l) ->
      let sk = ControlFlow.Table (Utils.s4 min_val, Utils.s4 max_val) in
      let e = ControlFlow.Switch_edge (sk, def, l) in
      ControlFlow.add_edge acc v e lbl)
    g
    changes_edges

let dummy_evaluation = PartialEvaluation.make_empty ()

type evaluation = {
    lines : int32 * Utils.u2 list;
    mutable evaluation_start : PartialEvaluation.t;
    mutable evaluation_end : PartialEvaluation.t;
    mutable operations : (Utils.u2 * PartialEvaluation.operation) list;
  }

let update_evaluation n e =
  let stack = ref e.PartialEvaluation.stack in
  for _i = 1 to n do
    stack := PartialEvaluation.pop !stack
  done;
  { e with PartialEvaluation.stack = !stack }

let equal_evaluation x y =
  if x == dummy_evaluation || y == dummy_evaluation then
    false
  else
    PartialEvaluation.equal x y

let optimize_partial_jumps g =
  let rec last_two = function
    | [] -> []
    | hd :: [] -> [hd]
    | hd1 :: hd2 :: [] -> [hd2; hd1]
    | _ :: tl -> last_two tl in
  let rec remove_last_one acc = function
    | [] -> assert false
    | _ :: [] -> List.rev acc
    | hd :: tl -> remove_last_one (hd :: acc) tl in
  let remove_last_one l = remove_last_one [] l in
  let rec remove_last_two acc = function
    | []
    | _ :: [] -> assert false
    | _ :: _ :: [] -> List.rev acc
    | hd :: tl -> remove_last_two (hd :: acc) tl in
  let remove_last_two l = remove_last_two [] l in
  let changes_edges, changes_handlers =
    ControlFlow.fold_vertices
      (fun v (acc_edges, acc_handlers) ->
        let acc_edges =
          try
            let v_lbl = ControlFlow.label_of_vertex v in
            let e_lbl, e = ControlFlow.edge_of_vertex g v in
            let int_comp two c d1 d2 lbl =
              let d = if c then d1 else d2 in
              v_lbl.operations <- (if two then remove_last_two else remove_last_one) v_lbl.operations;
              let e' = ControlFlow.Unconditional_edge (not c, d) in
              (v, e', lbl) :: acc_edges in
            match e, last_two v_lbl.operations with (* last two are in reversed order *)
            | ControlFlow.Unconditional_edge (_, dst),
              (_, PartialEvaluation.Push_long (x, _)) :: _
              when (ControlFlow.instructions_of_vertex dst) = [ Instruction.L2I ] ->
                let e2_lbl, e2 = ControlFlow.edge_of_vertex g dst in
                (match e2 with
                | ControlFlow.Conditional_edge (ControlFlow.Integer_equal_zero, ifso, ifno) ->
                    int_comp false (x = 0L) ifso ifno e2_lbl
                | ControlFlow.Conditional_edge (ControlFlow.Integer_greater_or_equal_zero, ifso, ifno) ->
                    int_comp false (x >= 0L) ifso ifno e2_lbl
                | ControlFlow.Conditional_edge (ControlFlow.Integer_greater_zero, ifso, ifno) ->
                    int_comp false (x > 0L) ifso ifno e2_lbl
                | ControlFlow.Conditional_edge (ControlFlow.Integer_lower_or_equal_zero, ifso, ifno) ->
                    int_comp false (x <= 0L) ifso ifno e2_lbl
                | ControlFlow.Conditional_edge (ControlFlow.Integer_lower_zero, ifso, ifno) ->
                    int_comp false (x < 0L) ifso ifno e2_lbl
                | ControlFlow.Conditional_edge (ControlFlow.Integer_not_equal_zero, ifso, ifno) ->
                    int_comp false (x <> 0L) ifso ifno e2_lbl
                | _ ->
                    acc_edges)
            | ControlFlow.Conditional_edge (ControlFlow.References_equal, dest, _),
              [(_, PartialEvaluation.Push_null _); (_, PartialEvaluation.Push_null _)] ->
                v_lbl.operations <- remove_last_two v_lbl.operations;
                let e' = ControlFlow.Unconditional_edge (false, dest) in
                (v, e', e_lbl) :: acc_edges
            | ControlFlow.Conditional_edge (ControlFlow.References_not_equal, _, dest),
              [(_, PartialEvaluation.Push_null _); (_, PartialEvaluation.Push_null _)] ->
                v_lbl.operations <- remove_last_two v_lbl.operations;
                let e' = ControlFlow.Unconditional_edge (true, dest) in
                (v, e', e_lbl) :: acc_edges
            | ControlFlow.Conditional_edge (ControlFlow.Integers_equal, ifso, ifno),
              [(_, PartialEvaluation.Push_integer (y, _)); (_, PartialEvaluation.Push_integer (x, _))] ->
                int_comp true (x = y) ifso ifno e_lbl
            | ControlFlow.Conditional_edge (ControlFlow.Integers_greater_or_equal, ifso, ifno),
              [(_, PartialEvaluation.Push_integer (y, _)); (_, PartialEvaluation.Push_integer (x, _))] ->
                int_comp true (x >= y) ifso ifno e_lbl
            | ControlFlow.Conditional_edge (ControlFlow.Integers_greater, ifso, ifno),
              [(_, PartialEvaluation.Push_integer (y, _)); (_, PartialEvaluation.Push_integer (x, _))] ->
                int_comp true (x > y) ifso ifno e_lbl
            | ControlFlow.Conditional_edge (ControlFlow.Integers_lower_or_equal, ifso, ifno),
              [(_, PartialEvaluation.Push_integer (y, _)); (_, PartialEvaluation.Push_integer (x, _))] ->
                int_comp true (x <= y) ifso ifno e_lbl
            | ControlFlow.Conditional_edge (ControlFlow.Integers_lower, ifso, ifno),
              [(_, PartialEvaluation.Push_integer (y, _)); (_, PartialEvaluation.Push_integer (x, _))] ->
                int_comp true (x < y) ifso ifno e_lbl
            | ControlFlow.Conditional_edge (ControlFlow.Integers_not_equal, ifso, ifno),
              [(_, PartialEvaluation.Push_integer (y, _)); (_, PartialEvaluation.Push_integer (x, _))] ->
                int_comp true (x <> y) ifso ifno e_lbl
            | ControlFlow.Conditional_edge (ControlFlow.Integer_equal_zero, ifso, ifno),
              (_, PartialEvaluation.Push_integer (x, _)) :: _ ->
                int_comp false (x = 0l) ifso ifno e_lbl
            | ControlFlow.Conditional_edge (ControlFlow.Integer_greater_or_equal_zero, ifso, ifno),
              (_, PartialEvaluation.Push_integer (x, _)) :: _ ->
                int_comp false (x >= 0l) ifso ifno e_lbl
            | ControlFlow.Conditional_edge (ControlFlow.Integer_greater_zero, ifso, ifno),
              (_, PartialEvaluation.Push_integer (x, _)) :: _ ->
                int_comp false (x > 0l) ifso ifno e_lbl
            | ControlFlow.Conditional_edge (ControlFlow.Integer_lower_or_equal_zero, ifso, ifno),
              (_, PartialEvaluation.Push_integer (x, _)) :: _ ->
                int_comp false (x <= 0l) ifso ifno e_lbl
            | ControlFlow.Conditional_edge (ControlFlow.Integer_lower_zero, ifso, ifno),
              (_, PartialEvaluation.Push_integer (x, _)) :: _ ->
                int_comp false (x < 0l) ifso ifno e_lbl
            | ControlFlow.Conditional_edge (ControlFlow.Integer_not_equal_zero, ifso, ifno),
              (_, PartialEvaluation.Push_integer (x, _)) :: _ ->
                int_comp false (x <> 0l) ifso ifno e_lbl
            | ControlFlow.Conditional_edge (ControlFlow.Reference_null, dest, _),
              (_, PartialEvaluation.Push_null _) :: _ ->
                v_lbl.operations <- remove_last_one v_lbl.operations;
                let e' = ControlFlow.Unconditional_edge (false, dest) in
                (v, e', e_lbl) :: acc_edges
            | ControlFlow.Conditional_edge (ControlFlow.Reference_not_null, _, dest),
              (_, PartialEvaluation.Push_null _) :: _ ->
                v_lbl.operations <- remove_last_one v_lbl.operations;
                let e' = ControlFlow.Unconditional_edge (true, dest) in
                (v, e', e_lbl) :: acc_edges
            | ControlFlow.Switch_edge (ControlFlow.Table (low, high), default, dests),
              (_, PartialEvaluation.Push_integer (x, _)) :: _ ->
                v_lbl.operations <- remove_last_one v_lbl.operations;
                let low = (low :> int32) in
                let high = (high :> int32) in
                let dest =
                  if (x >= low) && (x <= high) then
                    let idx = Int32.to_int (Int32.sub x low) in
                    List.nth dests idx
                  else
                    default in
                let e' = ControlFlow.Unconditional_edge (false, dest) in
                (v, e', e_lbl) :: acc_edges
            | ControlFlow.Switch_edge (ControlFlow.Lookup cases, default, dests),
              (_, PartialEvaluation.Push_integer (x, _)) :: _ ->
                v_lbl.operations <- remove_last_one v_lbl.operations;
                let l = List.combine (cases :> int32 list) dests in
                let dest =
                  try
                    snd (List.find (fun (c, _) -> c = x) l)
                  with Not_found -> default in
                let e' = ControlFlow.Unconditional_edge (false, dest) in
                (v, e', e_lbl) :: acc_edges
            | _ -> acc_edges
          with Not_found -> acc_edges in
        acc_edges, acc_handlers)
      g
      ([], []) in
  if (changes_edges = []) && (changes_handlers = []) then
    g
  else begin
    let g =
      List.fold_left
        (fun acc (v, e, lbl) -> ControlFlow.add_edge acc v e lbl)
        g
        changes_edges in
    List.fold_left
      (fun acc (v, l) ->
        ControlFlow.set_handlers acc v l)
      g
      changes_handlers
  end

let optimize_partial_evaluation ev g =
  let unify x y =
    if x == dummy_evaluation then
      y
    else if y == dummy_evaluation then
      x
    else
      PartialEvaluation.unify x y in
  let g =
    visit_graph
      ~container:Queue
      ~init_mark:(fun x ->
        { lines = x;
          evaluation_start = dummy_evaluation;
          evaluation_end = dummy_evaluation;
          operations = []; })
      ~init_root:(fun v ->
        let lbl = ControlFlow.label_of_vertex v in
        lbl.evaluation_start <- ev)
      ~visit_node:(fun pending_add _ vertex edge handlers _ ->
        let mark v e =
          let lbl = ControlFlow.label_of_vertex v in
          let e' = unify lbl.evaluation_start e in
          if not (equal_evaluation lbl.evaluation_start e') then begin
            lbl.evaluation_start <- e';
            pending_add v
          end in
        let label = ControlFlow.label_of_vertex vertex in
        let instructions = ControlFlow.instructions_of_vertex vertex in
        let initial_evaluation = label.evaluation_start in
        let empty_stack = PartialEvaluation.empty () in
        let cumulated_evaluation = { initial_evaluation with PartialEvaluation.stack = empty_stack } in
        let is_protected = handlers <> [] in
        let final_evaluation, cumulated_evaluation, operations =
          List.fold_left2
            (fun (evaluation, cumulated, operations) instr line ->
              let new_evaluation, operation = PartialEvaluation.update instr evaluation in
              let cumulated =
                if is_protected then
                  unify
                    cumulated
                    { new_evaluation with PartialEvaluation.stack = empty_stack }
                else
                  cumulated in
              (new_evaluation, cumulated, (line, operation) :: operations))
            (initial_evaluation, cumulated_evaluation, [])
            instructions
            (snd label.lines) in
        let operations = List.rev operations in
        (match edge with
        | Some (ControlFlow.Unconditional_edge (_, dst)) ->
            label.evaluation_end <- final_evaluation;
            label.operations <- operations;
            mark dst final_evaluation
        | Some (ControlFlow.Conditional_edge (jk, ifso, ifno)) ->
            let final_evaluation = update_evaluation (ControlFlow.nb_args_of_jump_kind jk) final_evaluation in
            label.evaluation_end <- final_evaluation;
            label.operations <- operations;
            mark ifso final_evaluation;
            mark ifno final_evaluation
        | Some (ControlFlow.Switch_edge (sk, dest, dests)) ->
            let final_evaluation = update_evaluation (ControlFlow.nb_args_of_switch_kind sk) final_evaluation in
            label.evaluation_end <- final_evaluation;
            label.operations <- operations;
            mark dest final_evaluation;
            List.iter (fun x -> mark x final_evaluation) dests
        | None ->
            label.operations <- operations);
        List.iter
          (fun (_, _, dst) ->
            let state = { cumulated_evaluation with PartialEvaluation.stack = PartialEvaluation.only_exception () } in
            mark dst state)
          handlers)
      g in
  let g = Utils.fix_point (==) optimize_partial_jumps g in
  let g =
    ControlFlow.map_graph
      (fun x _ ->
        let compiled = PartialEvaluation.map_operations x.operations in
        let compiled = Peephole.optimize_list compiled in
        let lines, instructions = List.split compiled in
        (fst x.lines, lines), instructions)
      (fun x _ -> x)
      (fun x _ _ -> x)
      g in
  (* jump are optimized because optimize_partial_jumps may have
     created some empty nodes *)
  optimize_jumps g

type forced = Forced_condition | Forced_sequence | Forced_weaksequence | Not_forced

type 'a block = {
    mutable id : int32;
    mutable in_line : bool;
    mutable additional_goto_target : int32 option;
    mutable forced : forced;
    mutable invert_condition : bool;
    lines : Utils.u2 list;
    old_label : 'a;
  }

type flattening_infos = {
    target_offset : int32;
    goto_offset : int32 option;
  }

let offset_of_flattening_infos fi =
  fi.target_offset

let flattened_graph g =
  map_graph
    (fun (x, y) ->
      { target_offset = x; goto_offset = None; }, y)
    g

module Int32Set = Set.Make (Int32)

let flatten_graph_with_goto_size ?(use_offsets = false) wide g =
  let goto_size = if wide then 5 else 3 in
  let blocks = ref [] in
  let next_id () = Int32.of_int (List.length !blocks) in
  let total_order =
    try
      let _ =
        ControlFlow.fold_vertices
          (fun v already_seen ->
            let ofs, _ = ControlFlow.label_of_vertex v in
            if Int32Set.mem ofs already_seen then
              raise Already_present
            else
              Int32Set.add ofs already_seen)
          g
          Int32Set.empty in
      true
    with Already_present -> false in
  let compare_by_ofs x y =
    let x = ControlFlow.label_of_vertex x in
    let y = ControlFlow.label_of_vertex y in
    Pervasives.compare x.old_label y.old_label in
  let use_offsets = use_offsets && total_order in
  let g =
    if use_offsets then begin
      let res =
        map_graph
          (fun (x, y) ->
            { id = Int32.min_int;
              in_line = false;
              additional_goto_target = None;
              forced = Not_forced;
              invert_condition = false;
              lines = y;
              old_label = x; })
          g in
      ControlFlow.iter_vertices
        (fun v ->
          blocks := v :: !blocks)
        res;
      blocks := List.sort compare_by_ofs !blocks;
      let counter = ref Int32.zero in
      List.iter
        (fun v ->
          let lbl = ControlFlow.label_of_vertex v in
          lbl.id <- !counter;
          counter := Int32.succ !counter)
        !blocks;
      res
    end else begin
      (* we delay the marking of return nodes with no handler *)
      let delayed = ref [] in
      let terminal v graph =
        let instrs = ControlFlow.instructions_of_vertex v in
        let handlers = ControlFlow.handlers_of_vertex graph v in
        let contains_return =
          List.exists
            (function
              | Instruction.ARETURN
              | Instruction.DRETURN
              | Instruction.FRETURN
              | Instruction.IRETURN
              | Instruction.LRETURN
              | Instruction.RETURN -> true
              | _ -> false)
            instrs in
        contains_return && (handlers = []) in
      let g =
        visit_graph
          ~container:Modifiable_stack
          ~init_mark:(fun (x, y) ->
            { id = Int32.min_int;
              in_line = false;
              additional_goto_target = None;
              forced = Not_forced;
              invert_condition = false;
              lines = y;
              old_label = x; })
          ~init_root:(fun v ->
            let lbl = ControlFlow.label_of_vertex v in
            lbl.id <- 0l;
            blocks := v :: !blocks)
          ~visit_node:(fun pending_add pending_add_bottom vertex edge handlers graph ->
            let mark v =
              let lbl = ControlFlow.label_of_vertex v in
              if lbl.id = Int32.min_int then begin
                if terminal v graph then begin
                  delayed := v :: !delayed;
                  false
                end else begin
                  lbl.id <- next_id ();
                  blocks := v :: !blocks;
                  true
                end
              end else
                false in
            let update_ids from increment =
              List.iter
                (fun b ->
                  let lbl = ControlFlow.label_of_vertex b in
                  let update = if increment then Int32.succ else Int32.pred in
                  if lbl.id >= from then begin
                    lbl.id <- update lbl.id
                  end)
                !blocks in
            let force_id f wanted_id v =
              let lbl = ControlFlow.label_of_vertex v in
              let ofs_v = lbl.old_label in
              let ofs_vertex = (ControlFlow.label_of_vertex vertex).old_label in
              if (f < lbl.forced) && (lbl.id > 0l) && (ofs_v > ofs_vertex) then begin
                let current_id = lbl.id in
                let next_id = next_id () in
                if current_id = wanted_id then begin
                  ()
                end else if wanted_id < next_id then begin
                  update_ids current_id false;
                  update_ids wanted_id true;
                  lbl.id <- wanted_id
                end else if wanted_id = next_id then begin
                  update_ids current_id false;
                  lbl.id <- Int32.pred wanted_id
                end;
                lbl.forced <- f;
                true
              end else
                false in
            let mark_list l =
              let res = ref [] in
              let l =
                List.sort
                  (fun x y ->
                    let ofs_x : int32 = (ControlFlow.label_of_vertex x).old_label in
                    let ofs_y : int32 = (ControlFlow.label_of_vertex y).old_label in
                    Pervasives.compare ofs_x ofs_y)
                  l in
              List.iter
                (fun x ->
                  let marked = mark x in
                  if marked then res := x :: !res)
                l;
              List.rev !res in
            (match edge with
            | Some (ControlFlow.Unconditional_edge (true, dst)) ->
                let marked = mark dst in
                if marked then pending_add dst;
                let lbl = ControlFlow.label_of_vertex vertex in
                let wanted_id = Int32.succ lbl.id in
                ignore (force_id Forced_sequence wanted_id dst)
            | Some (ControlFlow.Unconditional_edge (false, dst)) ->
                let marked = mark dst in
                if marked then pending_add dst;
                let lbl = ControlFlow.label_of_vertex vertex in
                let wanted_id = Int32.succ lbl.id in
                ignore (force_id Forced_weaksequence wanted_id dst)
            | Some (ControlFlow.Conditional_edge (_, ifso, ifno)) ->
                let marked_ifno = mark ifno in
                let marked_ifso = mark ifso in
                if marked_ifso then pending_add ifso;
                if marked_ifno then pending_add ifno;
                let lbl = ControlFlow.label_of_vertex vertex in
                let wanted_id = Int32.succ lbl.id in
                let forced_ifno = force_id Forced_condition wanted_id ifno in
                if not forced_ifno then begin
                  let forced_ifso = force_id Forced_condition wanted_id ifso in
                  if forced_ifso then lbl.invert_condition <- true
                end
            | Some (ControlFlow.Switch_edge (_, dest, dests)) ->
                let l = dest :: dests in
                let marked = mark_list l in
                List.iter pending_add_bottom marked
            | None -> ());
            let handler_dests = List.map (fun (_, _, dst) -> dst) handlers in
            let marked = mark_list handler_dests in
            List.iter pending_add_bottom marked)
          g in
      List.iter
        (fun v ->
          let lbl = ControlFlow.label_of_vertex v in
          if lbl.id = Int32.min_int then begin (* the same block may appear several times *)
            lbl.id <- next_id ();
            blocks := v :: !blocks
          end)
        (List.sort compare_by_ofs !delayed);
      g
    end in
  let compare_by_id x y =
    let x = ControlFlow.label_of_vertex x in
    let y = ControlFlow.label_of_vertex y in
    Pervasives.compare y.id x.id in
  let blocks = List.sort compare_by_id !blocks in
  (* check which blocks are in-line *)
  List.iter
    (fun vertex ->
      let lbl = ControlFlow.label_of_vertex vertex in
      let edge = try Some (snd (ControlFlow.edge_of_vertex g vertex)) with Not_found -> None in
      match edge with
      | Some (ControlFlow.Unconditional_edge (_, dst)) ->
          lbl.in_line <- (Int32.succ lbl.id) = (ControlFlow.label_of_vertex dst).id
      | Some (ControlFlow.Conditional_edge (_, _, ifno)) when not lbl.invert_condition ->
          lbl.in_line <- (Int32.succ lbl.id) = (ControlFlow.label_of_vertex ifno).id;
      | Some (ControlFlow.Conditional_edge (_, ifso, _)) when lbl.invert_condition ->
          lbl.in_line <- (Int32.succ lbl.id) = (ControlFlow.label_of_vertex ifso).id;
      | Some (ControlFlow.Conditional_edge (_, _, _)) ->
          lbl.in_line <- false
      | Some (ControlFlow.Switch_edge (_, _, _)) ->
          lbl.in_line <- false
      | None -> ())
    blocks;
  (* compute offsets *)
  let _, blocks =
    List.fold_right
      (fun v (ofs, acc) ->
        let lbl = ControlFlow.label_of_vertex v in
        let instrs = ControlFlow.instructions_of_vertex v in
        let edge = try Some (snd (ControlFlow.edge_of_vertex g v)) with Not_found -> None in
        let size = Instruction.size_of_list ofs instrs in
        let size = size + 
          match edge with
          | Some (ControlFlow.Unconditional_edge _) ->
              if lbl.in_line then 0 else goto_size
          | Some (ControlFlow.Conditional_edge (_, _, _)) ->
              (if wide then 3 + 5 else 3) + (if lbl.in_line then 0 else goto_size)
          | Some (ControlFlow.Switch_edge (sk, _, l)) ->
              let switch_ofs = ofs + size in
              1 + (3 - (switch_ofs mod 4)) + 4 + 4 +
                (match sk with
                | ControlFlow.Table _ -> 4 + 4 * (List.length l)
                | ControlFlow.Lookup _ -> 8 * (List.length l))
          | None -> 0 in
        lbl.id <- Int32.of_int ofs;
        ofs + size, v :: acc)
      blocks
      (0, []) in
  let blocks = List.rev blocks in
  (* add suffixes *)
  let instructions =
    List.map
      (fun v ->
        let lbl = ControlFlow.label_of_vertex v in
        let instrs = ControlFlow.instructions_of_vertex v in
        let size = Int32.of_int (Instruction.size_of_list (Int32.to_int lbl.id) instrs) in
        let ofs_of_vertex32 x k =
          Int32.sub
            (ControlFlow.label_of_vertex x).id
            (Int32.add (Int32.add lbl.id size) k) in
        let ofs_of_vertex x k =
          let x = ofs_of_vertex32 x k in
          if x >= -32768l && x <= 32767l then
            Int32.to_int x
          else
            invalid_arg "BaristaLibrary.Code.flatten_graph" in
        let edge = try Some (snd (ControlFlow.edge_of_vertex g v)) with Not_found -> None in
        match edge with
        | Some (ControlFlow.Unconditional_edge (_, dst)) ->
            if lbl.in_line then
              instrs
            else if wide then
              instrs @ [ Instruction.GOTO_W (Utils.s4 (ofs_of_vertex32 dst 0l)) ]
            else
              instrs @ [ Instruction.GOTO (Utils.s2 (ofs_of_vertex dst 0l)) ]
        | Some (ControlFlow.Conditional_edge (jk, ifso, ifno)) ->
            let jk, ifso, ifno =
              if lbl.invert_condition then
                ControlFlow.opposite_jump_kind jk, ifno, ifso
              else
                jk, ifso, ifno in
            if wide then begin
              if not lbl.in_line then
                lbl.additional_goto_target <- Some (Int32.add lbl.id (Int32.add size 8l));
              let next = Utils.s2 (3 + goto_size) in
              instrs
              @ (match ControlFlow.opposite_jump_kind jk with
              | ControlFlow.References_equal -> [ Instruction.IF_ACMPEQ next ]
              | ControlFlow.References_not_equal -> [ Instruction.IF_ACMPNE next ]
              | ControlFlow.Integers_equal -> [ Instruction.IF_ICMPEQ next ]
              | ControlFlow.Integers_greater_or_equal -> [ Instruction.IF_ICMPGE next ]
              | ControlFlow.Integers_greater -> [ Instruction.IF_ICMPGT next ]
              | ControlFlow.Integers_lower_or_equal -> [ Instruction.IF_ICMPLE next ]
              | ControlFlow.Integers_lower -> [ Instruction.IF_ICMPLT next ]
              | ControlFlow.Integers_not_equal -> [ Instruction.IF_ICMPNE next ]
              | ControlFlow.Integer_equal_zero -> [ Instruction.IFEQ next ]
              | ControlFlow.Integer_greater_or_equal_zero -> [ Instruction.IFGE next ]
              | ControlFlow.Integer_greater_zero -> [ Instruction.IFGT next ]
              | ControlFlow.Integer_lower_or_equal_zero -> [ Instruction.IFLE next ]
              | ControlFlow.Integer_lower_zero -> [ Instruction.IFLT next ]
              | ControlFlow.Integer_not_equal_zero -> [ Instruction.IFNE next ]
              | ControlFlow.Reference_null -> [ Instruction.IFNULL next ]
              | ControlFlow.Reference_not_null -> [ Instruction.IFNONNULL next ])
              @ ([ Instruction.GOTO_W (Utils.s4 (ofs_of_vertex32 ifso 3l)) ])
              @ (if lbl.in_line then
                []
              else
                [ Instruction.GOTO_W (Utils.s4 (ofs_of_vertex32 ifno 8l)) ])
            end else begin
              let ofs = Utils.s2 (ofs_of_vertex ifso 0l) in
              instrs
              @ (match jk with
              | ControlFlow.References_equal -> [ Instruction.IF_ACMPEQ ofs ]
              | ControlFlow.References_not_equal -> [ Instruction.IF_ACMPNE ofs ]
              | ControlFlow.Integers_equal -> [ Instruction.IF_ICMPEQ ofs ]
              | ControlFlow.Integers_greater_or_equal -> [ Instruction.IF_ICMPGE ofs ]
              | ControlFlow.Integers_greater -> [ Instruction.IF_ICMPGT ofs ]
              | ControlFlow.Integers_lower_or_equal -> [ Instruction.IF_ICMPLE ofs ]
              | ControlFlow.Integers_lower -> [ Instruction.IF_ICMPLT ofs ]
              | ControlFlow.Integers_not_equal -> [ Instruction.IF_ICMPNE ofs ]
              | ControlFlow.Integer_equal_zero -> [ Instruction.IFEQ ofs ]
              | ControlFlow.Integer_greater_or_equal_zero -> [ Instruction.IFGE ofs ]
              | ControlFlow.Integer_greater_zero -> [ Instruction.IFGT ofs ]
              | ControlFlow.Integer_lower_or_equal_zero -> [ Instruction.IFLE ofs ]
              | ControlFlow.Integer_lower_zero -> [ Instruction.IFLT ofs ]
              | ControlFlow.Integer_not_equal_zero -> [ Instruction.IFNE ofs ]
              | ControlFlow.Reference_null -> [ Instruction.IFNULL ofs ]
              | ControlFlow.Reference_not_null -> [ Instruction.IFNONNULL ofs ])
              @ (if lbl.in_line then
                []
              else
                [ Instruction.GOTO (Utils.s2 (ofs_of_vertex ifno 3l)) ])
            end
          | Some (ControlFlow.Switch_edge (sk, default, l)) ->
              instrs
              @ (match sk with
              | ControlFlow.Table (low, high) ->
                  [ Instruction.TABLESWITCH (Utils.s4 (ofs_of_vertex32 default 0l),
                                             low,
                                             high,
                                             List.map (fun x -> Utils.s4 (ofs_of_vertex32 x 0l)) l) ]
              | ControlFlow.Lookup k ->
                  [ Instruction.LOOKUPSWITCH (Utils.s4 (ofs_of_vertex32 default 0l),
                                              Utils.s4 (Int32.of_int (List.length l)),
                                              List.map2 (fun x y -> x, Utils.s4 (ofs_of_vertex32 y 0l)) k l) ])
          | None -> instrs)
      blocks in
  (* build line number table *)
  let offset_line_couples =
    List.map
      (fun v ->
        let lbl = ControlFlow.label_of_vertex v in
        let instrs = ControlFlow.instructions_of_vertex v in
        let _, res =
          List.fold_left2
            (fun (ofs, lst) instr line ->
              let size = Instruction.size_of ofs instr in
              (ofs + size, (ofs, line) :: lst))
            (Int32.to_int lbl.id, [])
            instrs
            lbl.lines in
        List.rev res)
      blocks in
  let offset_line_couples = List.flatten offset_line_couples in
  let _, offset_line_couples =
    List.fold_left
      (fun (prev_line, acc) (ofs, line) ->
        match prev_line with
        | Some prev when line <> prev ->
            (Some line, (Utils.u2 ofs, line) :: acc)
        | _ ->
            (prev_line, acc))
      (None, [])
      offset_line_couples in
  (* build exception table *)
  let table : (int * int * int * Name.for_class option) list list =
    ControlFlow.fold_handlers_per_vertice
      (fun v _ cn v' acc ->
        let ofs = Int32.to_int (ControlFlow.label_of_vertex v).id in
        let ofs' = Int32.to_int (ControlFlow.label_of_vertex v').id in
        let i = ControlFlow.instructions_of_vertex v in
        let sz = Instruction.size_of_list ofs i in
        if sz > 0 then
          (ofs, ofs + sz, ofs', cn) :: acc
        else
          acc)
      g
      [] in
  let table : (int * int * int * Name.for_class option) list list =
    List.filter (fun l -> l <> []) table in
  let table : (int * int * (int * Name.for_class option) list) list =
    List.map
      (function
        | ((ofs_start, ofs_end, _, _) :: _) as l ->
            ofs_start, ofs_end, List.rev_map (fun (_, _, d, x) -> d, x) l
        | [] ->
            assert false)
      table in
  let table = List.sort Pervasives.compare table in
  let same_class cn1 cn2 =
    match cn1, cn2 with
    | (Some n1), (Some n2) -> Name.equal_for_class n1 n2
    | None, None -> true
    | Some _, None | None, Some _ -> false in
  let rec same_list l l' =
    match l, l' with
    | (hd_dest, hd_class) :: tl,
      (hd_dest', hd_class') :: tl' ->
        (hd_dest = hd_dest') && (same_class hd_class hd_class') && (same_list tl tl')
    | [], [] -> true
    | _ :: _, []
    | [], _ :: _ -> false in
  let rec optimize_table = function
    | (ofs_start, ofs_end, l) :: (ofs_start', ofs_end', l') :: tl
      when ofs_end = ofs_start' && same_list l l' ->
        optimize_table ((ofs_start, ofs_end', l) :: tl)
    | hd :: tl ->
        hd :: (optimize_table tl)
    | [] ->
        [] in
  let table : (int * int * (int * Name.for_class option) list) list =
    optimize_table table in
  let table =
    List.map
      (fun (ofs_start, ofs_end, l) ->
        List.map
          (fun (ofs_dest, cn) ->
            { Attribute.try_start = Utils.u2 ofs_start;
              Attribute.try_end = Utils.u2 ofs_end;
              Attribute.catch = Utils.u2 ofs_dest;
              Attribute.caught = cn; })
          l)
      table in
  (List.flatten instructions),
  (List.rev offset_line_couples),
  (List.concat table),
  map_graph
    (fun x ->
      { target_offset = x.id; goto_offset = x.additional_goto_target; }, x.old_label)
    g

let flatten_graph ?(use_offsets = false) g =
  try
    flatten_graph_with_goto_size ~use_offsets false g
  with _ ->
    flatten_graph_with_goto_size ~use_offsets true g

let optimize_graph ?(rules=Peephole.all_rules) g =
  let g = Peephole.optimize_graph ~rules g in
  let g = optimize_jumps g in
  let g = optimize_switches g in
  let g = remove_dead_code g in
  g

let dummy_state = StackState.make_empty ()

type mark = {
    offset : int32;
    flattening_infos : flattening_infos;
    mutable stack_state : StackState.t;
    mutable stack_state_end : StackState.t;
    mutable max_stack : int;
    mutable max_locals : int;
  }

let java_lang_Throwable = Name.make_for_class_from_external @"java.lang.Throwable"

let update_state ofs n s =
  let stack = ref s.StackState.stack in
  for _i = 1 to n do
    stack := StackState.pop ofs !stack
  done;
  { s with StackState.stack = !stack }

let equal_state x y =
  if x == dummy_state || y == dummy_state then
     false
  else
    StackState.equal x y

module IntMap = Map.Make (struct
  type t = int
  let compare (x : int) (y : int) = Pervasives.compare x y
end)

let compute_stack_infos class_name unify_fun g s =
  let initial_max_locals = StackState.locals_size s in
  let unify x y =
    if x == dummy_state then
      y
    else if y == dummy_state then
      x
    else
      StackState.unify unify_fun x y in
  let g =
    visit_graph
      ~container:Queue
      ~init_mark:(fun (x, _) ->
        { offset = x.target_offset;
          flattening_infos = x;
          stack_state = dummy_state;
          stack_state_end = dummy_state;
          max_stack = 0;
          max_locals = 0; })
      ~init_root:(fun v ->
        let lbl = ControlFlow.label_of_vertex v in
        lbl.stack_state <- s)
      ~visit_node:(fun pending_add _ vertex edge handlers _ ->
        let mark v s =
          let lbl = ControlFlow.label_of_vertex v in
          let s' = unify lbl.stack_state s in
          if not (equal_state lbl.stack_state s') then begin
            lbl.stack_state <- s';
            lbl.max_locals <- 0;
            lbl.max_stack <- 0;
            pending_add v
          end in
        let label = ControlFlow.label_of_vertex vertex in
        let instructions = ControlFlow.instructions_of_vertex vertex in
        let initial_state = label.stack_state in
        let offset = Int32.to_int label.offset in
        let empty_stack = StackState.empty () in
        let cumulated_state = { initial_state with StackState.stack = empty_stack } in
        let is_protected = handlers <> [] in
        let final_state, cumulated_state, _ =
          List.fold_left
            (fun (state, cumulated, ofs) instr ->
              let sz = Instruction.size_of ofs instr in
              let new_state = StackState.update class_name (Utils.u2 ofs) instr state in
              let cumulated =
                if is_protected then
                  unify
                    cumulated
                    { new_state with StackState.stack = empty_stack }
                else
                  cumulated in
              label.max_stack <- Utils.max_int label.max_stack (StackState.stack_size new_state);
              label.max_locals <- Utils.max_int label.max_locals (StackState.locals_size new_state);
              (new_state, cumulated, ofs + sz))
            (initial_state, cumulated_state, offset)
            instructions in
        (match edge with
        | Some (ControlFlow.Unconditional_edge (_, dst)) ->
            label.stack_state_end <- final_state;
            mark dst final_state
        | Some (ControlFlow.Conditional_edge (jk, ifso, ifno)) ->
            let final_state = update_state (Utils.u2 offset) (ControlFlow.nb_args_of_jump_kind jk) final_state in
            label.stack_state_end <- final_state;
            mark ifso final_state;
            mark ifno final_state
        | Some (ControlFlow.Switch_edge (sk, dest, dests)) ->
            let final_state = update_state (Utils.u2 offset) (ControlFlow.nb_args_of_switch_kind sk) final_state in
            label.stack_state_end <- final_state;
            mark dest final_state;
            List.iter (fun x -> mark x final_state) dests
        | None -> ());
        List.iter
          (fun (_, cn, dst) ->
            let cn = match cn with Some x -> x | None -> java_lang_Throwable in
            let state = { cumulated_state with StackState.stack = StackState.only_exception cn } in
            mark dst state)
          handlers)
      g in
  let max_stack, max_locals, frames =
    ControlFlow.fold_vertices
      (fun v (ms, ml, sf) ->
        let lbl = ControlFlow.label_of_vertex v in
        let ofs = Int32.to_int lbl.offset in
        (Utils.max_int ms lbl.max_stack,
         Utils.max_int ml lbl.max_locals,
         (if (lbl.stack_state != dummy_state) then
           try
             let old = IntMap.find ofs sf in
             IntMap.add ofs (unify old lbl.stack_state) sf
           with Not_found ->
             IntMap.add ofs lbl.stack_state sf
         else
           sf)))
      g
      (0, 0, IntMap.empty) in
  let max_locals' =
    ControlFlow.fold_vertices
      (fun v acc ->
        let instrs = ControlFlow.instructions_of_vertex v in
        List.fold_left
          (fun acc i ->
            try
              let index =
                match i with
                | Instruction.ALOAD p -> Utils.u2_of_u1 p
                | Instruction.ALOAD_0 -> Utils.u2 0
                | Instruction.ALOAD_1 -> Utils.u2 1
                | Instruction.ALOAD_2 -> Utils.u2 2
                | Instruction.ALOAD_3 -> Utils.u2 3
                | Instruction.ASTORE p -> Utils.u2_of_u1 p
                | Instruction.ASTORE_0 -> Utils.u2 0
                | Instruction.ASTORE_1 -> Utils.u2 1
                | Instruction.ASTORE_2 -> Utils.u2 2
                | Instruction.ASTORE_3 -> Utils.u2 3
                | Instruction.DLOAD p -> Utils.u2_succ (Utils.u2_of_u1 p)
                | Instruction.DLOAD_0 -> Utils.u2 1
                | Instruction.DLOAD_1 -> Utils.u2 2
                | Instruction.DLOAD_2 -> Utils.u2 3
                | Instruction.DLOAD_3 -> Utils.u2 4
                | Instruction.DSTORE p -> Utils.u2_succ (Utils.u2_of_u1 p)
                | Instruction.DSTORE_0 -> Utils.u2 1
                | Instruction.DSTORE_1 -> Utils.u2 2
                | Instruction.DSTORE_2 -> Utils.u2 3
                | Instruction.DSTORE_3 -> Utils.u2 4
                | Instruction.FLOAD p -> Utils.u2_of_u1 p
                | Instruction.FLOAD_0 -> Utils.u2 0
                | Instruction.FLOAD_1 -> Utils.u2 1
                | Instruction.FLOAD_2 -> Utils.u2 2
                | Instruction.FLOAD_3 -> Utils.u2 3
                | Instruction.FSTORE p -> Utils.u2_of_u1 p
                | Instruction.FSTORE_0 -> Utils.u2 0
                | Instruction.FSTORE_1 -> Utils.u2 1
                | Instruction.FSTORE_2 -> Utils.u2 2
                | Instruction.FSTORE_3 -> Utils.u2 3
                | Instruction.ILOAD p -> Utils.u2_of_u1 p
                | Instruction.ILOAD_0 -> Utils.u2 0
                | Instruction.ILOAD_1 -> Utils.u2 1
                | Instruction.ILOAD_2 -> Utils.u2 2
                | Instruction.ILOAD_3 -> Utils.u2 3
                | Instruction.ISTORE p -> Utils.u2_of_u1 p
                | Instruction.ISTORE_0 -> Utils.u2 0
                | Instruction.ISTORE_1 -> Utils.u2 1
                | Instruction.ISTORE_2 -> Utils.u2 2
                | Instruction.ISTORE_3 -> Utils.u2 3
                | Instruction.LLOAD p -> Utils.u2_succ (Utils.u2_of_u1 p)
                | Instruction.LLOAD_0 -> Utils.u2 1
                | Instruction.LLOAD_1 -> Utils.u2 2
                | Instruction.LLOAD_2 -> Utils.u2 3
                | Instruction.LLOAD_3 -> Utils.u2 4
                | Instruction.LSTORE p -> Utils.u2_succ (Utils.u2_of_u1 p)
                | Instruction.LSTORE_0 -> Utils.u2 1
                | Instruction.LSTORE_1 -> Utils.u2 2
                | Instruction.LSTORE_2 -> Utils.u2 3
                | Instruction.LSTORE_3 -> Utils.u2 4
                | Instruction.WIDE_ALOAD p -> p
                | Instruction.WIDE_ASTORE p -> p
                | Instruction.WIDE_DLOAD p -> Utils.u2_succ p
                | Instruction.WIDE_DSTORE p -> Utils.u2_succ p
                | Instruction.WIDE_FLOAD p -> p
                | Instruction.WIDE_FSTORE p -> p
                | Instruction.WIDE_ILOAD p -> p
                | Instruction.WIDE_ISTORE p -> p
                | Instruction.WIDE_LLOAD p -> Utils.u2_succ p
                | Instruction.WIDE_LSTORE p -> Utils.u2_succ p
                | _ -> raise Not_found in
              Utils.max_u2 (Utils.u2_succ index) acc
            with Not_found -> acc)
          acc
          instrs)
      g
      (Utils.u2 0) in
  let frames =
    ControlFlow.fold_vertices
      (fun v acc->
        let lbl = ControlFlow.label_of_vertex v in
        match lbl.flattening_infos.goto_offset with
        | Some ofs ->
            let ofs = Int32.to_int ofs in
            if not (IntMap.mem ofs acc) then
              IntMap.add ofs lbl.stack_state_end acc
            else
              acc
        | None -> acc)
      g
      frames in
  let frames = IntMap.fold (fun x y acc -> (Utils.u2 x, y) :: acc) frames [] in
  Utils.u2 max_stack,
  Utils.u2 (Utils.max_int max_locals (Utils.max_int (max_locals' :> int) initial_max_locals)),
  StackState.encode (List.rev frames)
