(*
 * This file is part of Barista.
 * Copyright (C) 2007-2014 Xavier Clerc.
 *
 * Barista is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Barista is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *)


IFNDEF USE_JDK THEN

(* Type *)

type t = {
    write_u1 : int -> unit;
    write_bytes_from : Bytes.t -> int -> int -> unit;
    flush : unit -> unit;
    close : unit -> unit;
  }


(* Exception *)

BARISTA_ERROR =
  | Unable_to_open_stream of (path : Path.t) ->
      Printf.sprintf "unable to open output stream (%S)"
        (Path.to_string path)
  | Unable_to_write_data ->
      "unable to write data"
  | Unable_to_close_stream ->
      "unable to close stream"


(* Constructors *)

external string_of_bytes : Bytes.t -> string =
  "%identity"

external string_of_path : Path.t -> string =
  "%identity"

let make_of_buffer buff =
  let opened = ref true in
  let write_u1 x =
    if !opened then
      ByteBuffer.add_byte buff x
    else
      fail Unable_to_write_data in
  let write_bytes_from str pos len =
    if !opened then
      ByteBuffer.add_bytes_from buff str pos len
    else
      fail Unable_to_write_data in
  let flush () =
    () in
  let close () =
    opened := false in
  { write_u1; write_bytes_from; flush; close }

let make_of_channel chan =
  set_binary_mode_out chan true;
  let write_u1 x =
    try
      output_byte chan x
    with _ ->
      fail Unable_to_write_data in
  let write_bytes_from str pos len =
    let str = string_of_bytes str in
    try
      output chan str pos len
    with _ ->
      fail Unable_to_write_data in
  let flush () =
    try
      flush chan
    with _ ->
      fail Unable_to_write_data in
  let close () =
    try
      close_out chan
    with _ ->
      fail Unable_to_close_stream in
  { write_u1; write_bytes_from; flush; close }

let make_of_path path =
  try
    path
    |> string_of_path
    |> open_out
    |> make_of_channel
  with _ ->
    fail (Unable_to_open_stream path)


(* Casts, to avoid bound checking *)

external unsafe_u1 : Utils.u1 -> int =
  "%identity"

external unsafe_u2 : Utils.u2 -> int =
  "%identity"

external unsafe_u4 : Utils.u4 -> int64 =
  "%identity"

external unsafe_s1 : Utils.s1 -> int =
  "%identity"

external unsafe_s2 : Utils.s2 -> int =
  "%identity"

external unsafe_s4 : Utils.s4 -> int32 =
  "%identity"

external unsafe_s8 : Utils.s8 -> int64 =
  "%identity"


(* Functions *)

let write_u1 stream x =
  stream.write_u1 (unsafe_u1 x)

let write_u2 stream x =
  let x = unsafe_u2 x in
  let x1 = (x lsr 8) land 0x000000FF in
  let x2 = x land 0x000000FF in
  stream.write_u1 x1;
  stream.write_u1 x2

let write_u4 stream x =
  let x = unsafe_u4 x in
  let open Int64 in
  let x1 = logand (shift_right_logical x 24) 0x000000FFL in
  let x2 = logand (shift_right_logical x 16) 0x000000FFL in
  let x3 = logand (shift_right_logical x 8) 0x000000FFL in
  let x4 = logand x 0x000000FFL in
  stream.write_u1 (to_int x1);
  stream.write_u1 (to_int x2);
  stream.write_u1 (to_int x3);
  stream.write_u1 (to_int x4)

let write_s1 stream x =
  let x = unsafe_s1 x in
  let x' = x land 0x000000FF in
  stream.write_u1 x'

let write_s2 stream x =
  let x = unsafe_s2 x in
  let x1 = (x lsr 8) land 0x000000FF in
  let x2 = x land 0x000000FF in
  stream.write_u1 x1;
  stream.write_u1 x2

let write_s4 stream x =
  let x = unsafe_s4 x in
  let open Int32 in
  let x1 = logand (shift_right_logical x 24) 0x000000FFl in
  let x2 = logand (shift_right_logical x 16) 0x000000FFl in
  let x3 = logand (shift_right_logical x 8) 0x000000FFl in
  let x4 = logand x 0x000000FFl in
  stream.write_u1 (to_int x1);
  stream.write_u1 (to_int x2);
  stream.write_u1 (to_int x3);
  stream.write_u1 (to_int x4)

let write_s8 stream x =
  let x = unsafe_s8 x in
  let open Int64 in
  let x1 = logand (shift_right_logical x 56) 0x00000000000000FFL in
  let x2 = logand (shift_right_logical x 48) 0x00000000000000FFL in
  let x3 = logand (shift_right_logical x 40) 0x00000000000000FFL in
  let x4 = logand (shift_right_logical x 32) 0x00000000000000FFL in
  let x5 = logand (shift_right_logical x 24) 0x00000000000000FFL in
  let x6 = logand (shift_right_logical x 16) 0x00000000000000FFL in
  let x7 = logand (shift_right_logical x 8) 0x00000000000000FFL in
  let x8 = logand x 0x00000000000000FFL in
  stream.write_u1 (to_int x1);
  stream.write_u1 (to_int x2);
  stream.write_u1 (to_int x3);
  stream.write_u1 (to_int x4);
  stream.write_u1 (to_int x5);
  stream.write_u1 (to_int x6);
  stream.write_u1 (to_int x7);
  stream.write_u1 (to_int x8)

let write_bytes_from stream bytes pos len =
  stream.write_bytes_from bytes pos len

let write_bytes stream bytes =
  stream.write_bytes_from bytes 0 (Bytes.length bytes)

external unsafe_u2' : int -> Utils.u2 =
  "%identity"

let write_utf8 stream str =
  let bytes = UTF8.to_bytes str in
  write_u2 stream (unsafe_u2' (Bytes.length bytes));
  write_bytes stream bytes

let write_elements e stream f l =
  let len = List.length l in
  if len <= Utils.max_u2_value then begin
    write_u2 stream (Utils.u2 len);
    List.iter (fun x -> f stream x) l
  end else
    raise e

let flush stream =
  stream.flush ()

let close stream =
  stream.close ()

let close_noerr stream =
  try
    stream.close ()
  with _ ->
    ()

let try_with stream f =
  Utils.try_finally stream f close_noerr


(* Predefined streams *)

let stdout = make_of_channel stdout

let stderr = make_of_channel stderr

ELSE (* USE_JDK *)

(* Types *)

type t = java'io'DataOutputStream java_instance


(* Exception *)

BARISTA_ERROR =
  | Unable_to_open_stream of (path : Path.t) ->
      Printf.sprintf "unable to open output stream (%S)"
        (Path.to_string path)
  | Unable_to_write_data ->
      "unable to write data"
  | Unable_to_close_stream ->
      "unable to close stream"

(* Constructors *)

external byte_array_output_stream_of_byte_buffer : ByteBuffer.t -> java'io'ByteArrayOutputStream java_instance =
  "%identity"

external byte_array_of_bytes : Bytes.t -> int JavaByteArray.t =
  "%identity"

external output_stream_of_out_channel : out_channel -> java'io'OutputStream java_instance =
  "ocamljava_output_stream_of_out_channel"

external file_of_path : Path.t -> java'io'File java_instance =
  "%identity"

let make_of_buffer buff =
  buff
  |> byte_array_output_stream_of_byte_buffer
  |> Java.make "java.io.DataOutputStream(_)"

let make_of_channel chan =
  chan
  |> output_stream_of_out_channel
  |> Java.make "java.io.BufferedOutputStream(_)"
  |> Java.make "java.io.DataOutputStream(_)"

let make_of_path path =
  try
    path
    |> file_of_path
    |> Java.make "java.io.FileOutputStream(java.io.File)"
    |> Java.make "java.io.BufferedOutputStream(_)"
    |> Java.make "java.io.DataOutputStream(_)"
  with Java_exception _ ->
    fail (Unable_to_open_stream path)


(* Casts, to avoid bound checking *)

external unsafe_u1 : Utils.u1 -> int32 =
  "%int32_of_int"

external unsafe_u2 : Utils.u2 -> int32 =
  "%int32_of_int"

external unsafe_u4 : Utils.u4 -> int64 =
  "%identity"

external unsafe_s1 : Utils.s1 -> int32 =
  "%int32_of_int"

external unsafe_s2 : Utils.s2 -> int32 =
  "%int32_of_int"

external unsafe_s4 : Utils.s4 -> int32 =
  "%identity"

external unsafe_s8 : Utils.s8 -> int64 =
  "%identity"


(* Functions *)

let write_u1 stream x =
  try
    unsafe_u1 x
    |> Java.call "java.io.DataOutputStream.writeByte(_)" stream
  with Java_exception _ ->
    fail Unable_to_write_data

let write_u2 stream x =
  try
    unsafe_u2 x
    |> Java.call "java.io.DataOutputStream.writeShort(_)" stream
  with Java_exception _ ->
    fail Unable_to_write_data

let write_u4 stream x =
  try
    unsafe_u4 x
    |> Int64.to_int32
    |> Java.call "java.io.DataOutputStream.writeInt(_)" stream
  with Java_exception _ ->
    fail Unable_to_write_data

let write_s1 stream x =
  try
    unsafe_s1 x
    |> Java.call "java.io.DataOutputStream.writeByte(_)" stream
  with Java_exception _ ->
    fail Unable_to_write_data

let write_s2 stream x =
  try
    unsafe_s2 x
    |> Java.call "java.io.DataOutputStream.writeShort(_)" stream
  with Java_exception _ ->
    fail Unable_to_write_data

let write_s4 stream x =
  try
    unsafe_s4 x
    |> Java.call "java.io.DataOutputStream.writeInt(_)" stream
  with Java_exception _ ->
    fail Unable_to_write_data

let write_s8 stream x =
  try
    unsafe_s8 x
    |> Java.call "java.io.DataOutputStream.writeLong(_)" stream
  with Java_exception _ ->
    fail Unable_to_write_data

let write_bytes_from stream bytes pos len =
  try
    Java.call "java.io.DataOutputStream.write(byte[],_,_)"
      stream
      (byte_array_of_bytes bytes)
      (Int32.of_int pos)
      (Int32.of_int len)
  with Java_exception _ ->
    fail Unable_to_write_data

let write_bytes stream bytes =
  try
    Java.call "java.io.DataOutputStream.write(byte[])"
      stream
      (byte_array_of_bytes bytes)
  with Java_exception _ ->
    fail Unable_to_write_data

external java_string_of_utf8 : UTF8.t -> java'lang'String java_instance =
  "%identity"

let write_utf8 stream str =
  str
  |> java_string_of_utf8
  |>  Java.call "java.io.DataOutputStream.writeUTF(_)" stream

let write_elements e stream f l =
  let len = List.length l in
  if len <= Utils.max_u2_value then begin
    write_u2 stream (Utils.u2 len);
    List.iter (fun x -> f stream x) l
  end else
    raise e

let flush stream =
  try
    Java.call "java.io.DataOutputStream.flush()" stream
  with Java_exception _ ->
    fail Unable_to_write_data

let close stream =
  try
    Java.call "java.io.DataOutputStream.close()" stream
  with Java_exception _ ->
    fail Unable_to_close_stream

let close_noerr stream =
  try
    Java.call "java.io.DataOutputStream.close()" stream
  with Java_exception _ ->
    ()

let try_with stream f =
  Utils.try_finally stream f close_noerr


(* Predefined streams *)

let stdout = make_of_channel stdout

let stderr = make_of_channel stderr

END
