(*
 * This file is part of Barista.
 * Copyright (C) 2007-2014 Xavier Clerc.
 *
 * Barista is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Barista is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *)


open Consts

let (++) = UTF8.(++)


(* Base types and utility functions *)

BARISTA_ERROR =
  | Invalid_class_query of (x : UTF8.t) ->
      Printf.sprintf "invalid class query %S" (UTF8.to_string_noerr x)
  | Cannot_find_class of (x : UTF8.t) ->
      Printf.sprintf "cannot find class %S" (UTF8.to_string_noerr x)
  | Generic_arity_mismatch of (x : UTF8.t) * (y : int) * (z : int) ->
      Printf.sprintf "generic arity mismatch for class %S (%d found, but %d waited)"
        (UTF8.to_string_noerr x)
        y
        z
  | Generic_cannot_be_parametrized ->
      "a generic type variable cannot be parametrized"
  | Primitive_cannot_be_generic_argument ->
      "a primitive type cannot be used as a generic argument"
  | Wildcard_cannot_be_generic_argument ->
      "a wildcard type cannot be used as a generic argument"
  | Varargs_cannot_be_generic_argument ->
      "a varargs type cannot be used as a generic argument"
  | Generic_variable_bound_to_different_values of (x : UTF8.t) ->
      Printf.sprintf "generic variable %S is bound to different values"
        (UTF8.to_string_noerr x)
  | Void_cannot_be_an_array_element_type ->
      "void cannot be used as an array element type"
  | Invalid_generic_bound ->
      "invalid generic bound"
  | Ambiguity_in_generic_bound ->
      "ambiguity in generic bound (several classes match)"
  | No_class_match of (x : UTF8.t) ->
      Printf.sprintf "no class match for %S"
        (UTF8.to_string_noerr x)
  | Several_class_matches of (x : Name.for_class list) ->
      let x = List.map (fun x -> UTF8.to_string_noerr (Name.printable_utf8_for_class x)) x in
      Printf.sprintf "several class matches (%s)" (String.concat ", " x)
  | Invalid_field_query of (x : UTF8.t) ->
      Printf.sprintf "invalid field query %S" (UTF8.to_string_noerr x)
  | No_field_match of (x : UTF8.t) * (y : Name.for_field) ->
      Printf.sprintf "no field match for %S in %S"
        (UTF8.to_string_noerr (Name.utf8_for_field y))
        (UTF8.to_string_noerr x)
  | Several_field_matches of (x : (Name.for_class * Name.for_field) list) ->
      let x =
        List.map
          (fun x ->
            let cn = UTF8.to_string_noerr (Name.printable_utf8_for_class (fst x)) in
            let fn = UTF8.to_string_noerr (Name.utf8_for_field (snd x)) in
            cn ^ "." ^ fn)
          x in
      Printf.sprintf "several field matches (%s)" (String.concat ", " x)
  | Invalid_constructor_query of (x : UTF8.t) ->
      Printf.sprintf "invalid constructor query %S"
        (UTF8.to_string_noerr x)
  | No_constructor_match of (x : UTF8.t) ->
      Printf.sprintf "no constructor match for %S"
        (UTF8.to_string_noerr x)
  | Several_constructor_matches of (x : (Name.for_class * Descriptor.for_parameter list) list) ->
      let x =
        List.map
          (fun (cn, l) ->
            let cn = UTF8.to_string_noerr (Name.printable_utf8_for_class cn) in
            let l = List.map Descriptor.external_utf8_of_java_type (l :> Descriptor.java_type list) in
            let l = List.map UTF8.to_string_noerr l in
            cn ^ "(" ^ (String.concat ", " l) ^ ")")
          x in
      Printf.sprintf "several constructor matches (%s)"
        (String.concat ", " x)
  | Invalid_regular_method_query of (x : UTF8.t) ->
      Printf.sprintf "invalid regular method query %S"
        (UTF8.to_string_noerr x)
  | No_regular_method_match of (x : UTF8.t) * (y : Name.for_method) ->
      Printf.sprintf "no regular method match for %S in %S"
        (UTF8.to_string_noerr (Name.utf8_for_method y))
        (UTF8.to_string_noerr x)
  | Several_regular_method_matches of (x : (Name.for_class * Name.for_method * Descriptor.for_method) list) ->
      let x =
        List.map
          (fun (cn, mn, (l, r)) ->
            let cn = UTF8.to_string_noerr (Name.printable_utf8_for_class cn) in
            let mn = UTF8.to_string_noerr (Name.utf8_for_method mn) in
            let l = List.map Descriptor.external_utf8_of_java_type (l :> Descriptor.java_type list) in
            let l = List.map UTF8.to_string_noerr l in
            let r = UTF8.to_string_noerr (Descriptor.external_utf8_of_java_type r) in
            cn ^ "." ^ mn ^ "(" ^ (String.concat ", " l) ^ "):" ^ r)
          x in
      Printf.sprintf "several regular method matches (%s)" (String.concat ", " x)
  | Cannot_use_wide_wildcard_for_field_type ->
      "cannot use wide wildcard for field type"
  | Wide_wildcard_cannot_be_combined_with_another_parameter_type ->
      "wide wildcard cannot be combined with another parameter type"
  | Varargs_can_only_appear_at_the_end_of_a_type ->
      "varargs can only appear at the end of a type"

type java_type =
  [ `Boolean
  | `Byte
  | `Char
  | `Double
  | `Float
  | `Int
  | `Long
  | `Short
  | `Void
  | `Class of ClassDefinition.t
  | `Array of 'a
  | `Varargs of 'a
  | `Generic of UTF8.t
  | `Wildcard of bool] constraint 'a = non_void_java_type
and non_void_java_type =
  [ `Boolean
  | `Byte
  | `Char
  | `Double
  | `Float
  | `Int
  | `Long
  | `Short
  | `Class of ClassDefinition.t
  | `Array of 'a
  | `Varargs of 'a
  | `Generic of UTF8.t
  | `Wildcard of bool ] constraint 'a = non_void_java_type

type substitution = java_type UTF8.Map.t

type 'a result = {
    value : 'a;
    with_generics : bool;
    substitution : substitution;
  }

let rec equal_java_type x y =
  match x, y with
  | `Boolean, `Boolean -> true
  | `Byte, `Byte -> true
  | `Char, `Char -> true
  | `Double, `Double -> true
  | `Float, `Float -> true
  | `Int, `Int -> true
  | `Long, `Long -> true
  | `Short, `Short -> true
  | `Void, `Void -> true
  | `Class cd1, `Class cd2 ->
      let cn1 = cd1.ClassDefinition.name in
      let cn2 = cd2.ClassDefinition.name in
      Name.equal_for_class cn1 cn2
  | `Array a1, `Array a2 -> equal_java_type (a1 :> java_type) (a2 :> java_type)
  | `Varargs va1, `Varargs va2 -> equal_java_type (va1 :> java_type) (va2 :> java_type)
  | `Generic id1, `Generic id2 -> UTF8.equal id1 id2
  | `Wildcard w1, `Wildcard w2 -> w1 = w2
  | _ -> false

let rec matches_descriptor subst x y =
  match x, y with
  | `Boolean, `Boolean -> true
  | `Byte, `Byte -> true
  | `Char, `Char -> true
  | `Double, `Double -> true
  | `Float, `Float -> true
  | `Int, `Int -> true
  | `Long, `Long -> true
  | `Short, `Short -> true
  | `Void, `Void -> true
  | `Class cd1, `Class cn2 ->
      let cn1 = cd1.ClassDefinition.name in
      Name.equal_for_class cn1 cn2
  | `Array a1, `Array a2 ->
      matches_descriptor
        subst
        (a1 :> java_type)
        (a2 :> Descriptor.java_type)
  | `Varargs _, _ -> false
  | `Generic id, _ ->
      (try
        matches_descriptor subst (UTF8.Map.find id subst) y
      with Not_found -> true)
  | `Wildcard _, _ -> true
  | _ -> false


(* Support for classes *)

type for_class = {
    class_with_generics : bool;
    class_name : UTF8.t;
    class_parameters : for_class list;
    class_array_dimensions : int * bool;
  }

(* separators, as a string:
   - space;
   - tabulation;
   - lower_than;
   - greater_than;
   - opening_square_bracket;
   - comma;
   - opening_parenthesis;
   - closing_parenthesis. *)
let separators = @" \t<>[,()"

let rec read_class_desc generics state =
  let buf = UTF8Buffer.make () in
  state#consume_whitespace;
  while state#is_available && not (state#look_ahead_string separators) do
    UTF8Buffer.add_char buf state#consume_char
  done;
  state#consume_whitespace;
  let params = ref [] in
  if generics && state#is_available && (state#look_ahead lower_than) then begin
    params := read_class_desc_list generics state lower_than greater_than;
  end;
  state#consume_whitespace;
  let dims = ref 0 in
  while state#is_available && state#look_ahead opening_square_bracket do
    state#consume_only opening_square_bracket;
    state#consume_whitespace;
    state#consume_only closing_square_bracket;
    incr dims;
    state#consume_whitespace;
  done;
  state#consume_whitespace;
  let name = UTF8Buffer.contents buf in
  let name, varargs_before =
    if UTF8.ends_with @"..." name then
      let len = UTF8.length name in
      UTF8.substring name 0 (len - 4), true
    else
      name, false in
  let varargs_after =
    if state#is_available && state#look_ahead dot then begin
      state#consume_only dot;
      state#consume_only dot;
      state#consume_only dot;
      state#consume_whitespace;
      true
    end else
      false in
  if (varargs_before && varargs_after) || (varargs_before && (!dims > 0)) then
    fail Varargs_can_only_appear_at_the_end_of_a_type;
  { class_with_generics = generics;
    class_name = name;
    class_parameters = List.rev !params;
    class_array_dimensions = !dims, (varargs_before || varargs_after) }
and read_class_desc_list generics state open_char end_char =
  let res = ref [] in
  state#consume_only open_char;
  state#consume_whitespace;
  while not (state#look_ahead end_char) do
    if !res <> [] then begin
      state#consume_only comma;
      state#consume_whitespace;
    end;
    res := (read_class_desc generics state) :: !res;
    state#consume_whitespace;
  done;
  state#consume_only end_char;
  List.rev !res

let rec write_class_desc buf c =
  UTF8Buffer.add_string buf c.class_name;
  if c.class_parameters <> [] then begin
    UTF8Buffer.add_char buf lower_than;
    let first = ref true in
    List.iter
      (fun p ->
        if !first then
          first := false
        else
          UTF8Buffer.add_string buf @", ";
        write_class_desc buf p)
      c.class_parameters;
    UTF8Buffer.add_char buf greater_than
  end;
  for _i = 1 to fst c.class_array_dimensions do
    UTF8Buffer.add_string buf @"[]"
  done

type candidate =
  | ClassDef of ClassDefinition.t
  | Generic of UTF8.t

let rec check_bound loader class_bound (t : java_type) (s : Signature.field_type_signature) =
  let get_parent cd =
    match cd.ClassDefinition.extends with
    | Some parent ->
        (try
          `Class (ClassLoader.find_class_name loader parent)
        with _ ->
          fail (Cannot_find_class (Name.external_utf8_for_class parent)))
    | None -> fail Invalid_generic_bound in
  match t, s with
  | `Boolean, _
  | `Byte, _
  | `Char, _
  | `Double, _
  | `Float, _
  | `Int, _
  | `Long, _
  | `Short, _
  | `Void, _ -> fail Primitive_cannot_be_generic_argument
  | `Class _, Signature.Type_variable_signature _ -> ()
  | `Class cd, Signature.Class_type_signature cts ->
      if class_bound then begin
        let same =
          Name.equal_for_class
            cd.ClassDefinition.name
            cts.Signature.qualified_class_name in
        if not same then
          check_bound loader true (get_parent cd) s
      end else begin
        let exists =
          List.exists
            (fun itf ->
              Name.equal_for_class itf cts.Signature.qualified_class_name)
            cd.ClassDefinition.implements in
        if not exists then
          check_bound loader false (get_parent cd) s
      end
  | `Array _, Signature.Class_type_signature cts ->
      let cn = Name.printable_utf8_for_class cts.Signature.qualified_class_name in
      if not (UTF8.equal cn @"java.lang.Object") then
        fail Invalid_generic_bound
  | `Array _, Signature.Type_variable_signature _ -> ()
  | `Array a, Signature.Array_type_signature ats ->
      (match a, ats with
      | `Class cd, Signature.Field_type_signature fts ->
          check_bound loader true (`Class cd) fts
      | `Boolean, Signature.Base_type `Boolean
      | `Byte, Signature.Base_type `Byte
      | `Char, Signature.Base_type `Char
      | `Double, Signature.Base_type `Double
      | `Float, Signature.Base_type `Float
      | `Int, Signature.Base_type `Int
      | `Long, Signature.Base_type `Long
      | `Short, Signature.Base_type `Short -> ()
      | _ -> fail Invalid_generic_bound)
  | `Wildcard _, _
  | `Generic _, _ -> ()
  | _ -> fail Invalid_generic_bound

let rec resolve_class_desc open_packages subst loader c =
  let make_array (n, e) t =
    let rec ma n t =
      if n = 0 then t else ma (pred n) (`Array t) in
    if e then `Varargs (ma n t) else ma n t in
  let return : non_void_java_type -> (java_type * substitution) list =
    fun t -> [(make_array c.class_array_dimensions t :> java_type), subst] in
  match UTF8.to_string c.class_name with
  | "boolean" -> return `Boolean
  | "byte" -> return `Byte
  | "char" -> return `Char
  | "double" -> return `Double
  | "float" -> return `Float
  | "int" -> return `Int
  | "long" -> return `Long
  | "short" -> return `Short
  | "void" ->
      if (fst c.class_array_dimensions > 0) || (snd c.class_array_dimensions) then
        fail Void_cannot_be_an_array_element_type;
      [`Void, subst]
  | "_" -> return (`Wildcard false)
  | "-" -> return (`Wildcard true)
  | _ ->
      let candidates =
        try
          [ ClassDef (ClassLoader.find_class loader c.class_name) ]
        with _ ->
          let rec find_all acc = function
            | package :: tl ->
                let name = package ++ @"." ++ c.class_name in
                (try
                  let x = ClassDef (ClassLoader.find_class loader name) in
                  find_all (x :: acc) tl
                with _ -> find_all acc tl)
            | [] ->
                if acc = [] && c.class_with_generics then
                  [Generic c.class_name]
                else
                  acc in
          find_all [] open_packages in
      if candidates = [] then fail (Cannot_find_class c.class_name);
      let len_effective = List.length c.class_parameters in
      List.map
        (function
          | Generic id ->
              if len_effective > 0 then
                fail Generic_cannot_be_parametrized;
              (make_array c.class_array_dimensions (`Generic id) :> java_type), subst
          | ClassDef candidate ->
              let subst = ref subst in
              if len_effective > 0 then begin
                let sign =
                  try
                    Attribute.extract_class_signature candidate.ClassDefinition.attributes
                  with Not_found -> fail (Generic_arity_mismatch (c.class_name, len_effective, 0)) in
                let len_formal = List.length sign.Signature.formal_type_parameters in
                if len_formal <> len_effective then
                  fail (Generic_arity_mismatch (c.class_name, len_effective, len_formal));
                List.iter2
                  (fun f e ->
                    let res = resolve_class_desc open_packages !subst loader e in
                    let res = match res with
                    | [x, _] -> x
                    | _ -> fail Ambiguity_in_generic_bound in
                    (match res with
                    | `Boolean
                    | `Byte
                    | `Char
                    | `Double
                    | `Float
                    | `Int
                    | `Long
                    | `Short
                    | `Void -> fail Primitive_cannot_be_generic_argument
                    | `Wildcard _ -> fail Wildcard_cannot_be_generic_argument
                    | `Varargs _ -> fail Varargs_cannot_be_generic_argument
                    | `Generic _ -> ()
                    | `Array _
                    | `Class _ ->
                        check_bound loader true (res :> java_type) f.Signature.class_bound;
                        List.iter (check_bound loader false (res :> java_type)) f.Signature.interface_bounds);
                    let id = f.Signature.identifier in
                    (try
                      let already_bound = UTF8.Map.find id !subst in
                      if not (equal_java_type (already_bound :> java_type) (res :> java_type)) then
                        fail (Generic_variable_bound_to_different_values id)
                    with Not_found -> ());
                    subst := UTF8.Map.add id res !subst)
                  sign.Signature.formal_type_parameters
                  c.class_parameters;
              end;
              (make_array c.class_array_dimensions (`Class candidate) :> java_type), !subst)
        candidates

let utf8_for_class c =
  let buf = UTF8Buffer.make () in
  write_class_desc buf c;
  UTF8Buffer.contents buf

let make_for_class_from_utf8 generics s =
  let state = new UTF8LexerState.t s in
  let res =
    try
      read_class_desc generics state
    with _ -> fail (Invalid_class_query s) in
  if state#is_available then
    fail (Invalid_class_query s)
  else
    res

let search_for_classes ?(open_packages = []) loader c =
  let res = resolve_class_desc open_packages UTF8.Map.empty loader c in
  List.fold_left
    (fun acc (elem, subst) ->
      match elem with
      | `Class cd ->
          let elem = { value = cd;
                       with_generics = c.class_with_generics;
                       substitution = subst; } in
          elem :: acc
      | _ -> acc)
    []
    res

let search_for_class ?(open_packages = []) loader c =
  let res = search_for_classes ~open_packages loader c in
  match res with
  | [] -> fail (No_class_match c.class_name)
  | [ x ] -> x
  | _ ->
      let l = List.map (fun r -> r.value.ClassDefinition.name) res in
      fail (Several_class_matches l)

let for_classes generics ?(open_packages = []) loader s =
  let query = make_for_class_from_utf8 generics s in
  search_for_classes ~open_packages loader query

let for_class generics ?(open_packages = []) loader s =
  let query = make_for_class_from_utf8 generics s in
  search_for_class ~open_packages loader query


(* Support for fields *)

type for_field = {
    field_with_generics : bool;
    field_class : for_class;
    field_name : Name.for_field;
    field_type : for_class;
  }

let utf8_for_field f =
  let buf = UTF8Buffer.make () in
  write_class_desc buf f.field_class;
  UTF8Buffer.add_char buf @'.';
  UTF8Buffer.add_string buf (Name.utf8_for_field f.field_name);
  UTF8Buffer.add_char buf @':';
  write_class_desc buf f.field_type;
  UTF8Buffer.contents buf

let make_for_field_from_utf8 generics s =
  try
    let len = UTF8.length s in
    let idx_colon = UTF8.index_from s 0 colon in
    let idx_dot = UTF8.rindex_from s idx_colon dot in
    let class_part = UTF8.substring s 0 (pred idx_dot) in
    let name_part = UTF8.substring s (succ idx_dot) (pred idx_colon) in
    let type_part = UTF8.substring s (succ idx_colon) (pred len) in
    let class_lexer = new UTF8LexerState.t class_part in
    let class_part = read_class_desc generics class_lexer in
    if class_lexer#is_available then fail (Invalid_field_query s);
    let name_part = Name.make_for_field (UTF8.trim name_part) in
    let type_lexer = new UTF8LexerState.t type_part in
    let type_part = read_class_desc generics type_lexer in
    if type_lexer#is_available then fail (Invalid_field_query s);
    { field_with_generics = generics;
      field_class = class_part;
      field_name = name_part;
      field_type = type_part }
  with
  | (Exception _) as e -> raise e
  | _ -> fail (Invalid_field_query s)

let matches_type_signature loader (t : java_type) (s : Signature.field_type_signature) =
  try
    check_bound loader true t s;
    true
  with Exception _ -> false

module ClassSet = Set.Make (struct
  type t = Name.for_class
  let compare x y = Name.compare_for_class x y
end)

module FieldSet = Set.Make (struct
  type t = Field.t * ClassDefinition.t
  let compare (x, _) (y, _) =
    if (x == y) || (Name.equal_for_field x.Field.name y.Field.name) then
      0
    else
      Field.compare x y
end)

let search_for_fields ?(open_packages = []) loader f =
  let res = search_for_class ~open_packages loader f.field_class in
  let classes = Hierarchy.all_parent_class_definitions true loader res.value in
  let fields =
    List.fold_left
      (fun acc declaring_class ->
        List.fold_left
          (fun acc field -> FieldSet.add (field, declaring_class) acc)
          acc
          declaring_class.ClassDefinition.fields)
      FieldSet.empty
      classes in
  let fields = FieldSet.elements fields in
  let types = resolve_class_desc open_packages res.substitution loader f.field_type in
  if List.exists (function `Wildcard true, _ -> true | _ -> false) types then
    fail Cannot_use_wide_wildcard_for_field_type;
  List.fold_left
    (fun acc (field, declaring_class) ->
      if Name.equal_for_field f.field_name field.Field.name then begin
        if f.field_with_generics then begin
          try
            let _, subst =
              List.find
                (fun (t, s) ->
                  try
                    matches_type_signature
                      loader
                      t
                      (Attribute.extract_field_signature field.Field.attributes)
                  with Not_found ->
                    matches_descriptor
                      s
                      t
                      (field.Field.descriptor :> Descriptor.java_type))
                types in
            { value = (declaring_class, field);
              with_generics = true;
              substitution = subst; } :: acc
          with _ ->
            acc
        end else begin
          let matches =
            List.exists
              (fun (t, _) ->
                matches_descriptor
                  UTF8.Map.empty
                  t
                  (field.Field.descriptor :> Descriptor.java_type))
              types in
          if matches then
            { value = (declaring_class, field);
              with_generics = false;
              substitution = UTF8.Map.empty; } :: acc
          else
            acc
        end
      end else
        acc)
    []
    fields

let search_for_field ?(open_packages = []) loader f =
  let res = search_for_fields ~open_packages loader f in
  match res with
  | [] -> fail (No_field_match (f.field_class.class_name, f.field_name))
  | [ x ] -> x
  | _ ->
      let l =
        List.map
          (fun r ->
            (fst r.value).ClassDefinition.name,
            (snd r.value).Field.name)
          res in
      fail (Several_field_matches l)

let for_fields generics ?(open_packages = []) loader s =
  let query = make_for_field_from_utf8 generics s in
  search_for_fields ~open_packages loader query

let for_field generics ?(open_packages = []) loader s =
  let query = make_for_field_from_utf8 generics s in
  search_for_field ~open_packages loader query


(* Utilities for methods *)

type for_method = {
    method_with_generics : bool;
    method_regular : bool;
    method_class : for_class;
    method_name : Name.for_method;
    method_parameter_types : for_class list;
    method_return_type : for_class;
  }

let utf8_for_method m =
  let buf = UTF8Buffer.make () in
  write_class_desc buf m.method_class;
  if m.method_regular then begin
    UTF8Buffer.add_char buf @'.';
    UTF8Buffer.add_string buf (Name.utf8_for_method m.method_name);
  end;
  UTF8Buffer.add_char buf @'(';
  List.iter
    (write_class_desc buf)
    m.method_parameter_types;
  UTF8Buffer.add_char buf @')';
  if m.method_regular then begin
    UTF8Buffer.add_char buf @':';
    write_class_desc buf m.method_return_type;
  end;
  UTF8Buffer.contents buf

module MethodSet = Set.Make (struct
  type t = Method.t * ClassDefinition.t
  let compare (m1, _) (m2, _) =
    if m1 == m2 then
      0
    else
      match m1, m2 with
      | (Method.Regular r1), (Method.Regular r2) ->
          let c = Name.compare_for_method r1.Method.name r2.Method.name in
          if c <> 0 then
            c
          else
            Descriptor.compare_for_method
              r1.Method.descriptor
              r2.Method.descriptor
      | (Method.Constructor c1), (Method.Constructor c2) ->
          Descriptor.compare_for_method
            (c1.Method.cstr_descriptor, `Void)
            (c2.Method.cstr_descriptor, `Void)
      | (Method.Initializer i1), (Method.Initializer i2) ->
          Method.compare_class_initializer i1 i2
      | _ -> Method.compare m1 m2
end)

let search_for_methods open_packages loader m =
  let res = search_for_class ~open_packages loader m.method_class in
  let methods =
    if m.method_regular then begin
      let classes = Hierarchy.all_parent_class_definitions true loader res.value in
      let interfaces, classes =
        List.partition
          (fun cd -> AccessFlag.mem_class `Interface cd.ClassDefinition.access_flags)
          classes in
      let classes = interfaces @ classes in
      let methods =
        List.fold_left
          (fun acc declaring_class ->
            List.fold_left
              (fun acc meth -> MethodSet.add (meth, declaring_class) acc)
              acc
              declaring_class.ClassDefinition.methods)
          MethodSet.empty
          classes in
      MethodSet.elements methods
    end else
      List.map (fun m -> m, res.value) res.value.ClassDefinition.methods in
  let matches_descriptor_any candidates desc =
    List.exists
      (fun (t, _) ->
        matches_descriptor
          UTF8.Map.empty
          t
          (desc :> Descriptor.java_type))
      candidates in
  let matches_descriptor_any_generics candidates desc =
    match desc with
    | Signature.Field_type_signature fts ->
        List.exists
          (fun (t, _) ->
            matches_type_signature
              loader
              t
              fts)
          candidates
    | Signature.Base_type bt ->
        matches_descriptor_any candidates bt in
  let eff_params =
    List.map
      (fun x ->
        resolve_class_desc open_packages res.substitution loader x)
      m.method_parameter_types in
  let matches_everything =
    List.fold_left
      (fun acc l ->
        let l : (java_type * substitution) list = l in
        let contains_wide_wildcard =
          List.exists (function `Wildcard true, _ -> true | _ -> false) l in
        if contains_wide_wildcard && (List.length eff_params > 1) then
          fail Wide_wildcard_cannot_be_combined_with_another_parameter_type;
        acc || contains_wide_wildcard)
      false
      eff_params in
  let matches_method_descriptor varargs ((params, ret) : Descriptor.for_method) attrs =
    let eff_params =
      if varargs then begin
        let patch_varargs l =
          match List.rev l with
          | (`Varargs va, subst) :: tl -> ((`Array va, subst) :: tl) |> List.rev
          | _ -> l in
        List.map patch_varargs eff_params
      end else
        eff_params in
    let len_eff = List.length m.method_parameter_types in
    let len_desc = List.length params in
    let without_generics () =
      ((not m.method_regular)
     || (matches_descriptor_any
           (resolve_class_desc open_packages res.substitution loader m.method_return_type)
           ret))
        &&
      (matches_everything
     ||
      ((len_eff = len_desc)
        &&
      (List.for_all2
         (fun eff desc ->
           matches_descriptor_any
             eff
             desc)
         eff_params
         params))) in
      if m.method_with_generics then begin
        try
          let sign = Attribute.extract_method_signature attrs in
          let len_desc = List.length sign.Signature.types in
          ((not m.method_regular)
         || (matches_descriptor_any_generics
               (resolve_class_desc open_packages res.substitution loader m.method_return_type)
               sign.Signature.return))
            &&
          (matches_everything
         || ((len_eff = len_desc)
            &&
          (List.for_all2
             (fun eff desc ->
               matches_descriptor_any_generics
                 eff
                 desc)
             eff_params
             sign.Signature.types)))
        with Not_found ->
          without_generics ()
      end else (* m.method_with_generics *)
        without_generics () in
  List.fold_left
    (fun acc (meth, declaring_class) ->
      let add_acc () =
        { value = (declaring_class, meth);
          with_generics = m.method_with_generics;
          substitution = res.substitution } :: acc in
      match meth with
      | Method.Regular { Method.flags; name; descriptor; attributes; _ }
        when m.method_regular
            && (Name.equal_for_method name m.method_name)
            && (matches_method_descriptor
                  (AccessFlag.mem_method `Varargs flags)
                  descriptor
                  attributes) ->
            add_acc ()
      | Method.Constructor { Method.cstr_flags; cstr_descriptor; cstr_attributes; _ }
        when (not m.method_regular)
            && (matches_method_descriptor
                  (AccessFlag.mem_constructor `Varargs cstr_flags)
                  (cstr_descriptor, `Void)
                  cstr_attributes) ->
              add_acc ()
      | _ -> acc)
    []
    methods


(* Support for constructors *)

type for_constructor = for_method

let utf8_for_constructor c =
  utf8_for_method c

let make_for_constructor_from_utf8 generics s =
  try
    let len = UTF8.length s in
    let idx_open = UTF8.index_from s 0 opening_parenthesis in
    let idx_close = UTF8.index_from s idx_open closing_parenthesis in
    let class_part = UTF8.substring s 0 (pred idx_open) in
    let params_part = UTF8.substring s idx_open idx_close in
    let last_part = UTF8.substring s (succ idx_close) (pred len) in
    let class_lexer = new UTF8LexerState.t class_part in
    let class_part = read_class_desc generics class_lexer in
    let params_lexer = new UTF8LexerState.t params_part in
    let params_part = read_class_desc_list generics params_lexer opening_parenthesis closing_parenthesis in
    let last_lexer = new UTF8LexerState.t last_part in
    last_lexer#consume_whitespace;
    if last_lexer#is_available then fail (Invalid_constructor_query s);
    { method_with_generics = generics;
      method_regular = false;
      method_class = class_part;
      method_name = Name.make_for_method class_constructor;
      method_parameter_types = params_part;
      method_return_type = class_part }
  with
  | (Exception _) as e -> raise e
  | _ -> fail (Invalid_constructor_query s)

let search_for_constructors ?(open_packages = []) loader c =
  let res = search_for_methods open_packages loader c in
  List.map
    (fun x ->
      match snd x.value with
      | Method.Constructor c ->
          { value = (fst x.value), c;
            with_generics = x.with_generics;
            substitution = x.substitution }
      | _ -> assert false)
    res

let search_for_constructor ?(open_packages = []) loader c =
  let res = search_for_constructors ~open_packages loader c in
  match res with
  | [] -> fail (No_constructor_match c.method_class.class_name)
  | [ x ] -> x
  | _ ->
      let l =
        List.map
          (fun r ->
            (fst r.value).ClassDefinition.name,
            (snd r.value).Method.cstr_descriptor)
          res in
      fail (Several_constructor_matches l)

let for_constructors generics ?(open_packages = []) loader s =
  let query = make_for_constructor_from_utf8 generics s in
  search_for_constructors ~open_packages loader query

let for_constructor generics ?(open_packages = []) loader s =
  let query = make_for_constructor_from_utf8 generics s in
  search_for_constructor ~open_packages loader query


(* Support for regular methods *)

type for_regular_method = for_method

let utf8_for_regular_method rm =
  utf8_for_method rm

let make_for_regular_method_from_utf8 generics s =
  try
    let len = UTF8.length s in
    let idx_open = UTF8.index_from s 0 opening_parenthesis in
    let idx_close = UTF8.index_from s idx_open closing_parenthesis in
    let idx_dot = UTF8.rindex_from s idx_open dot in
    let class_part = UTF8.substring s 0 (pred idx_dot) in
    let name_part = UTF8.substring s (succ idx_dot) (pred idx_open) in
    let params_part = UTF8.substring s idx_open idx_close in
    let last_part = UTF8.substring s (succ idx_close) (pred len) in
    let class_lexer = new UTF8LexerState.t class_part in
    let class_part = read_class_desc generics class_lexer in
    let name_part = Name.make_for_method (UTF8.trim name_part) in
    let params_lexer = new UTF8LexerState.t params_part in
    let params_part = read_class_desc_list generics params_lexer opening_parenthesis closing_parenthesis in
    let last_lexer = new UTF8LexerState.t last_part in
    last_lexer#consume_whitespace;
    last_lexer#consume_only colon;
    last_lexer#consume_whitespace;
    let last_part = read_class_desc generics last_lexer in
    if last_lexer#is_available then fail (Invalid_regular_method_query s);
    { method_with_generics = generics;
      method_regular = true;
      method_class = class_part;
      method_name = name_part;
      method_parameter_types = params_part;
      method_return_type = last_part }
  with
  | (Exception _) as e -> raise e
  | _ -> fail (Invalid_regular_method_query s)

let search_for_regular_methods ?(open_packages = []) loader rm =
  let res = search_for_methods open_packages loader rm in
  List.map
    (fun x ->
      match snd x.value with
      | Method.Regular r ->
          { value = (fst x.value), r;
            with_generics = x.with_generics;
            substitution = x.substitution }
      | _ -> assert false)
    res

let search_for_regular_method ?(open_packages = []) loader rm =
  let res = search_for_regular_methods ~open_packages loader rm in
  match res with
  | [] -> fail (No_regular_method_match (rm.method_class.class_name, rm.method_name))
  | [ x ] -> x
  | _ ->
      let l =
        List.map
          (fun r ->
            (fst r.value).ClassDefinition.name,
            (snd r.value).Method.name,
            (snd r.value).Method.descriptor)
          res in
      fail (Several_regular_method_matches l)

let for_regular_methods generics ?(open_packages = []) loader s =
  let query = make_for_regular_method_from_utf8 generics s in
  search_for_regular_methods ~open_packages loader query

let for_regular_method generics ?(open_packages = []) loader s =
  let query = make_for_regular_method_from_utf8 generics s in
  search_for_regular_method ~open_packages loader query
