(*
 * This file is part of Barista.
 * Copyright (C) 2007-2014 Xavier Clerc.
 *
 * Barista is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Barista is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *)

open BaristaLibrary

let program_name =
  try
    OS.argv.(0)
    |> Path.make_of_utf8
    |> Path.basename
    |> UTF8.to_string
  with _ ->
    "barista"

let print_help commands =
  let print_usage, command_list =
    match commands with
    | Some l -> false, l
    | None -> true, Predefined.commands in

  if print_usage then
    Printf.eprintf "Usage: %s <command> <command arguments and switches...>\n\n" program_name;
  List.iter
    (fun (module Cmd : Command.T) ->
      let names =
        List.map
          (fun str -> Printf.sprintf "%S" (UTF8.to_string_noerr str))
          Cmd.names in
      let parameters = Cmd.make_parameters () in
      let switches = Cmd.switches_of_parameters parameters in
      let others = Cmd.others_of_parameters parameters in
      Printf.eprintf "command %s (%s)\n"
        (String.concat " / " names)
        (UTF8.to_string_noerr Cmd.description);
      (match others with
      | Some { UArg.argument_desc; _ } ->
          Printf.eprintf "  %s\n" (UTF8.to_string_noerr argument_desc)
      | None ->
          Printf.eprintf "  no argument should be provided\n");
      (match switches with
      | _ :: _ ->
          Printf.eprintf "  the following switches are recognized:\n";
          let switches_and_documentations =
            List.map
              (function { UArg.identifier; kind; documentation } ->
                let open UArg in
                let switch =
                  match kind with
                  | Nothing _ ->
                      Printf.sprintf "%s  "
                        (UTF8.to_string_noerr identifier)
                  | String { switch_desc; _ } ->
                      Printf.sprintf "%s %s  "
                        (UTF8.to_string_noerr identifier)
                        (UTF8.to_string_noerr switch_desc)
                  | Choice { switch_options; _ } ->
                      let options =
                        switch_options
                        |> List.map UTF8.to_string_noerr
                        |> String.concat " | " in                        
                      Printf.sprintf "%s { %s }  "
                        (UTF8.to_string_noerr identifier)
                        options in
                let documentation = UTF8.to_string_noerr documentation in
                switch, documentation)
              switches in
          let maximum_switch_length = 36 in
          List.iter
            (fun (switch, documentation) ->
                let len_switch = String.length switch in
                let len_padding = maximum_switch_length - len_switch in
                if len_padding >= 0 then begin
                  let padding = String.make (Utils.max_int 0 len_padding) ' ' in
                  Printf.eprintf "  %s%s%s\n" switch padding documentation
                end else begin
                  let padding = String.make maximum_switch_length ' ' in
                  Printf.eprintf "  %s\n  %s%s\n" switch padding documentation
                end)
            switches_and_documentations
      | [] ->
          Printf.eprintf "  no switch is recognized\n");
      Printf.eprintf "\n%!")
    command_list

let candidate_commands command =
  Utils.map_partial
    (fun ((module Cmd : Command.T) as cmd) ->
      try
        let name =
          List.find
            (fun n ->
              ((UTF8.length command) > 2)
                && (UTF8.starts_with command n))
            Cmd.names in
        Some (name, cmd)
      with Not_found ->
        None)
    Predefined.commands

let main () =
  if Array.length OS.argv > 1 then begin
    let command = OS.argv.(1) in
    match UTF8.to_string_noerr command with
    | "help" when Array.length OS.argv > 2 ->
        let sub_command = OS.argv.(2) in
        let commands = candidate_commands sub_command in
        (match commands with
        | [] ->
            Printf.eprintf "*** no command matches %S\n"
              (UTF8.to_string_noerr sub_command);
            exit 1
        | l ->
            print_help (Some (List.map snd l)))
    | "help" | "-help" | "--help" ->
        print_help None
    | command_as_string ->
        let commands = candidate_commands command in
        (match commands with
        | [] ->
            Printf.eprintf "*** no command matches %S\n" command_as_string;
            Printf.eprintf "*** type \"%s help\" to get the list of commands\n" program_name;
            exit 1
        | [_, (module Cmd : Command.T)] ->
            let params = Cmd.make_parameters () in
            UArg.parse
              (Cmd.switches_of_parameters params)
              (Cmd.others_of_parameters params)
              OS.argv
              2;
            Cmd.run params
        | _ ->
            Printf.eprintf "*** several commands match %S:\n" command_as_string;
            List.iter
              (fun (name, _) ->
                Printf.eprintf "  - %S\n" (UTF8.to_string_noerr name))
              commands;
            Printf.eprintf "*** type \"%s help\" to get the list of commands\n" program_name;
            exit 1)
    end

let () =
  try
    main ();
    exit 0
  with
  | Assembler.Exception (n, e) ->
      Printf.eprintf "*** assembler error line %d: %s\n" n (Assembler.string_of_error e);
      exit 1
  | FlowPrinter.Exception e ->
      Printf.eprintf "*** flow printer error: %s\n" (FlowPrinter.string_of_error e);
  | ClassPath.Exception e ->
      Printf.eprintf "*** class path error: %s\n" (ClassPath.string_of_error e);
      exit 1
  | ClassLoader.Exception e ->
      Printf.eprintf "*** class loader error: %s\n" (ClassLoader.string_of_error e);
      exit 1
  | Lookup.Exception e ->
      Printf.eprintf "*** lookup error: %s\n" (Lookup.string_of_error e);
      exit 1
  | Sys_error se ->
      Printf.eprintf "*** system error: %s\n" se;
      exit 1
  | e ->
      Printf.eprintf "*** error: %s\n" (Predefined.string_of_exception e);
      exit 1
