(*
 * This file is part of Barista.
 * Copyright (C) 2007-2014 Xavier Clerc.
 *
 * Barista is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Barista is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *)


(* Operators *)

let (+=) x i =
  x := !x + i

let (-=) x i =
    x := !x - i


(* Integer types *)

type integer_error = {
    type_name : string;
    lower_bound : int64;
    upper_bound : int64;
    value : int64;
  }

exception Integer_exception of integer_error

let fail_integer type_name lower_bound upper_bound value =
  raise (Integer_exception { type_name; lower_bound; upper_bound; value })

let fail_u1 value =
  fail_integer "u1" 0L 255L (Int64.of_int value)

let fail_s1 value =
  fail_integer "s1" (-128L) 127L (Int64.of_int value)

let fail_u2 value =
  fail_integer "u2" 0L 65535L (Int64.of_int value)

let fail_s2 value =
  fail_integer "s2" (-32768L) 32767L (Int64.of_int value)

let fail_u4 value =
  fail_integer "u4" 0L 4294967296L value

let fail_s4 value =
  fail_integer
    "s4"
    (Int64.of_int32 Int32.min_int)
    (Int64.of_int32 Int32.max_int)
    value

let fail_s8 value =
  fail_integer "s8" Int64.min_int Int64.max_int value

let string_of_integer_error ie =
  Printf.sprintf "%s value %Ld is out of bounds (%Ld..%Ld)"
    ie.type_name
    ie.value
    ie.lower_bound
    ie.upper_bound

let () =
  Printexc.register_printer
    (function
      | Integer_exception e -> Some (string_of_integer_error e)
      | _ -> None)

let min_int_value = Pervasives.min_int

let max_int_value = Pervasives.max_int

let min_int (x : int) (y : int) =
  if x < y then
    x
  else
    y

let max_int (x : int) (y : int) =
  if x > y then
    x
  else
    y

type u1 = int

let u1 x =
  if x >= 0 && x <= 255 then
    x
  else
    fail_u1 x

let min_u1_value = 0

let max_u1_value = 255

let min_u1 (x : u1) (y : u1) =
  if x < y then
    x
  else
    y

let max_u1 (x : u1) (y : u1) =
  if x > y then
    x
  else
    y

let u1_succ x =
  if x < 255 then
    succ x
  else
    fail_u1 (succ x)

let u1_pred x =
  if x > 0 then
    pred x
  else
    fail_u1 (pred x)

type s1 = int

let s1 x =
  if x >= -128 && x <= 127 then
    x
  else
    fail_s1 x

let min_s1_value = -128

let max_s1_value = 127

let min_s1 (x : s1) (y : s1) =
  if x < y then
    x
  else
    y

let max_s1 (x : s1) (y : s1) =
  if x > y then
    x
  else
    y

let s1_succ x =
  if x < 127 then
    succ x
  else
    fail_s1 (succ x)

let s1_pred x =
  if x > -128 then
    pred x
  else
    fail_s1 (pred x)

let s1_neg x =
  if x <> -128 then
    ~-x
  else
    fail_s1 (~-x)

type u2 = int

let u2 x =
  if x >= 0 && x <= 65535 then
    x
  else
    fail_u2 x

let min_u2_value = 0

let max_u2_value = 65535

let min_u2 (x : u2) (y : u2) =
  if x < y then
    x
  else
    y

let max_u2 (x : u2) (y : u2) =
  if x > y then
    x
  else
    y

let u2_succ x =
  if x < 65535 then
    succ x
  else
    fail_u2 (succ x)

let u2_pred x =
  if x > 0 then
    pred x
  else
    fail_u2 (pred x)

type s2 = int

let s2 x =
  if x >= -32768 && x <= 32767 then
    x
  else
    fail_s2 x

let min_s2_value = -32768

let max_s2_value = 32767

let min_s2 (x : s2) (y : s2) =
  if x < y then
    x
  else
    y

let max_s2 (x : s2) (y : s2) =
  if x > y then
    x
  else
    y

let s2_succ x =
  if x < 32767 then
    succ x
  else
    fail_s2 (succ x)

let s2_pred x =
  if x > -32768 then
    pred x
  else
    fail_s2 (pred x)

let s2_neg x =
  if x <> -32768 then
    ~-x
  else
    fail_s2 (~-x)

type u4 = int64

let u4 x =
  if x >= 0L && x <= 4294967296L then
    x
  else
    fail_u4 x

let min_u4_value = 0L

let max_u4_value = 4294967296L

let min_u4 (x : u4) (y : u4) =
  if x < y then
    x
  else
    y

let max_u4 (x : u4) (y : u4) =
  if x > y then
    x
  else
    y

let u4_succ x =
  if x < 4294967296L then
    Int64.succ x
  else
    fail_u4 (Int64.succ x)

let u4_pred x =
  if x > 0L then
    Int64.pred x
  else
    fail_u4 (Int64.pred x)

type s4 = int32

external s4 : int32 -> int32 =
  "%identity"

let min_s4_value = Int32.min_int

let max_s4_value = Int32.max_int

let min_s4 (x : s4) (y : s4) =
  if x < y then
    x
  else
    y

let max_s4 (x : s4) (y : s4) =
  if x > y then
    x
  else
    y

let s4_succ x =
  if x < Int32.max_int then
    Int32.succ x
  else
    fail_s4 (Int64.succ (Int64.of_int32 x))

let s4_pred x =
  if x > Int32.min_int then
    Int32.pred x
  else
    fail_s4 (Int64.pred (Int64.of_int32 x))

let s4_neg x =
  if x <> Int32.min_int then
    Int32.neg x
  else
    fail_s4 (Int64.neg (Int64.of_int32 x))

type s8 = int64

external s8 : int64 -> int64 =
  "%identity"

let min_s8_value = Int64.min_int

let max_s8_value = Int64.max_int

let min_s8 (x : s8) (y : s8) =
  if x < y then
    x
  else
    y

let max_s8 (x : s8) (y : s8) =
  if x > y then
    x
  else
    y

let s8_succ x =
  if x < Int64.max_int then
    Int64.succ x
  else
    fail_s8 (Int64.succ x)

let s8_pred x =
  if x > Int64.min_int then
    Int64.pred x
  else
    fail_s8 (Int64.pred x)

let s8_neg x =
  if x <> Int64.min_int then
    Int64.neg x
  else
    fail_s8 (Int64.neg x)

let split_s8 x =
  Int64.to_int32 (Int64.shift_right_logical x 32),
  Int64.to_int32 (Int64.logand x 0x00000000FFFFFFFFL)

let gather_s8 hi lo =
  Int64.logor
    (Int64.shift_left (Int64.of_int32 hi) 32)
    (Int64.logand (Int64.of_int32 lo) 0x00000000FFFFFFFFL)

external u2_of_u1 : u1 -> u2 =
  "%identity"

external s4_of_s2 : s2 -> s4 =
  "%int32_of_int"


(* List and array utilities *)

let rec list_mem eq x = function
  | hd :: tl -> (eq x hd) || (list_mem eq x tl)
  | [] -> false

let rec list_assoc eq x = function
  | (key, value) :: tl ->
      if eq key x then
        value
      else
        list_assoc eq x tl
  | [] -> raise Not_found

let rec map_partial f = function
  | hd :: tl ->
      (match f hd with
      | Some x -> x :: (map_partial f tl)
      | None -> map_partial f tl)
  | [] -> []

let map_list_to_array f l =
  match l with
  | hd :: tl ->
      let len = List.length l in
      let res = Array.make len (f hd) in
      let rec iter i = function
        | hd :: tl ->
            res.(i) <- f hd;
            iter (succ i) tl
        | [] -> res in
      iter 1 tl
  | [] -> [||]

let map_array_to_list f arr =
  let res = ref [] in
  for i = pred (Array.length arr) downto 0 do
    res := (f arr.(i)) :: !res;
  done;
  !res

let rec list_equal eq l1 l2 =
  (l1 == l2) ||
  (match l1, l2 with
  | (hd1 :: tl1), (hd2 :: tl2) ->
      if eq hd1 hd2 then list_equal eq tl1 tl2 else false
  | (_ :: _), [] -> false
  | [], (_ :: _) -> false
  | [], [] -> true)

let rec list_compare cmp l1 l2 =
  if l1 == l2 then
    0
  else
    match l1, l2 with
    | (hd1 :: tl1), (hd2 :: tl2) ->
        let res = cmp hd1 hd2 in
        if res = 0 then
          list_compare cmp tl1 tl2
        else
          res
    | (_ :: _), [] -> 1
    | [], (_ :: _) -> -1
    | [], [] -> 0

let list_hash h l =
  let rec iter acc rem = function
    | hd :: tl ->
        let acc = 17 * acc + (h hd) in
        let rem = pred rem in
        if rem = 0 then
          acc
        else
          iter acc rem tl
    | [] ->
        acc in
  iter 0 32 l

let array_equal eq arr1 arr2 =
  (arr1 == arr2) ||
  (let len = Array.length arr1 in
  if len = (Array.length arr2) then
    let i = ref 0 in
    while (!i < len) && (eq arr1.(!i) arr2.(!i)) do
      incr i
    done;
    !i = len
  else
    false)

let array_compare cmp arr1 arr2 =
  if arr1 == arr2 then
    0
  else
    let l1 = Array.length arr1 in
    let l2 = Array.length arr2 in
    if l1 = l2 then
      let rec iter i =
        if i < l1 then
          let res = cmp arr1.(i) arr2.(i) in
          if res = 0 then iter (succ i) else res
        else
          0 in
      iter 0
    else
      Pervasives.compare l1 l2

let array_hash h arr =
  let res = ref 0 in
  for i = 0 to min_int 31 (pred (Array.length arr)) do
    res := 17 * !res + (h arr.(i))
  done;
  !res


(* String utilities *)

let string_replace old_char new_char str =
  let res = String.copy str in
  let len = String.length res in
  for i = 0 to pred len do
    if res.[i] = old_char then res.[i] <- new_char
  done;
  res

let string_split seps str =
  let idx = ref 0 in
  let len = String.length str in
  let buff = Buffer.create len in
  let res = ref [] in
  let in_sep = ref false in
  while !idx < len do
    if !in_sep then begin
      if not (String.contains seps str.[!idx]) then begin
        Buffer.add_char buff str.[!idx];
        in_sep := false
      end
    end else begin
      if String.contains seps str.[!idx] then begin
        res := (Buffer.contents buff) :: !res;
        Buffer.clear buff;
        in_sep := true
      end else
        Buffer.add_char buff str.[!idx]
    end;
    incr idx
  done;
  let last = Buffer.contents buff in
  if last <> "" then res := last :: !res;
  List.rev !res

let string_of_list f l =
  match l with
  | hd :: tl ->
      let buff = Buffer.create 128 in
      Buffer.add_string buff "[";
      Buffer.add_string buff (f hd);
      let rec iter = function
        | h :: t ->
            Buffer.add_string buff "; ";
            Buffer.add_string buff (f h);
            iter t
        | [] ->
            () in
      iter tl;
      Buffer.add_string buff "]";
      Buffer.contents buff
  | [] ->
      "[]"

let string_of_array f arr =
  if arr = [||] then
    "[||]"
  else begin
    let buff = Buffer.create 128 in
    Buffer.add_string buff "[|";
    Buffer.add_string buff (f arr.(0));
    for i = 1 to pred (Array.length arr) do
      Buffer.add_string buff "; ";
      Buffer.add_string buff (f arr.(i))
    done;
    Buffer.add_string buff "|]";
    Buffer.contents buff
  end

let hash_string str =
  let res = ref 0 in
  for i = 0 to pred (String.length str) do
    res := 17 * !res + (Char.code str.[i])
  done;
  !res


(* Miscellaneous *)

let rec fix_point eq f x =
  let y = f x in
  if eq x y then
    y
  else
    fix_point eq f y

let compose_list l =
  fun x -> List.fold_left (fun acc f -> f acc) x l

let try_finally x f h =
  let res =
    try
      f x
    with e ->
      h x;
      raise e in
  h x;
  res

let identity x = x

let rec switch eq matches default x =
  match matches with
  | (key, func) :: tl ->
      if eq key x then
        func x
      else
        switch eq tl default x
  | [] ->
      default x

let universal_hash x =
  Hashtbl.hash x

let make_header_printer conv l =
  match l with
  | _ :: _ :: _ ->
      fun str ->
        let buff = Buffer.create 80 in
        Buffer.add_string buff "--- ";
        Buffer.add_string buff (conv str);
        Buffer.add_char buff ' ';
        for _i = 1 to 79 - (Buffer.length buff) do
          Buffer.add_char buff '-';
        done;
        print_endline (Buffer.contents buff)
  | _ ->
      fun _ -> ()
