(*
 * This file is part of Barista.
 * Copyright (C) 2007-2014 Xavier Clerc.
 *
 * Barista is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Barista is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *)

(** Definition of commands to be launched from command-line. *)


module type T = sig

  val names : UTF8.t list
  (** The various names for the command. *)

  val description : UTF8.t
  (** The short description used for command-line help. *)

  val latex_description : UTF8.t list
  (** The LaTeX-formatted description used for the manual. *)

  type parameters
  (** The type of parameters to be passed to the command. *)

  val make_parameters : unit -> parameters
  (** Builds parameters representing default values. *)

  val switches_of_parameters : parameters -> UArg.switch list
  (** Returns the command-line switches recognized by the command. *)

  val others_of_parameters : parameters -> UArg.others option
  (** Returns the function used to handle command-line elements that are
      not command-line switches or associated elements. *)

  val run : parameters -> unit
  (** Actually executes the command with passed parameters. *)

end
(** The type of commands with names, documentation, parameters, and
    implementation. *)

class no_parameter : object
  method switches : UArg.switch list
    (** Returns an empty list. *)
  method others : UArg.others option
    (** Returns [None]. *)
end
(** Class accepting no parameter. *)

class base_parameters : UTF8.t -> object
  method add_class_path : UTF8.t -> unit
    (** Adds the passed string to the class path. *)
  method set_class_path : UTF8.t -> unit
    (** Sets the class path to the passed value. *)
  method build_class_path : ClassPath.t
    (** Returns the class path. *)
  method add_element : UTF8.t -> unit
    (** Adds the passed value to the elements (unparsed command-line
        arguments). *)
  method get_elements : UTF8.t list
    (** Returns the elements (unparsed command-line arguments). *)
  method switches : UArg.switch list
    (** Returns the command-line switches handling class path. *)
  method others : UArg.others option
    (** Returns the function handling command-line elements that are not
        command-line switches or associated elements. *)
end
(** Class handling class path and unparsed elements.

    Class path can be modified through:
    - ["-cp ..."] to add elements to the class path;
    - ["-classpath ..."] to set class path. *)

class graph_format : object
  method set_graph_format : Graph.format -> unit
    (** Changes the graph format to the passed value. *)
  method get_graph_format : Graph.format
    (** Returns the graph format. *)
  method switches : UArg.switch list
    (** Returns the command-line switches handling graph format. *)
  method others : UArg.others option
    (** Returns [None]. *)
end
(** Class accepting handling graph format.

    Graph format can be modified through:
    - ["-format <f>"] to set the graph format. *)

val parameters_functions : (unit -> (< switches : UArg.switch list; others : UArg.others option; .. > as 'a)) -> (unit -> 'a) * ('a -> UArg.switch list) * ('a -> UArg.others option)
(** Returns functions suitable for [T.make_parameters],
    [T.switches_of_parameters], and [T.others_of_parameters] from a
    function building an object instance with methods [switches] and
    [others]. *)
