(*
 * This file is part of Barista.
 * Copyright (C) 2007-2014 Xavier Clerc.
 *
 * Barista is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Barista is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *)


let names = [ @"flow" ]

let description = @"prints the control flow of methods"

let latex_description = [
  @"Prints onto the standard output the control flow of the passed methods." ;
  @"Methods are represented by their signatures, leading to command-lines" ;
  @"akin to the following one:" ;
  @"\\begin{center}" ;
  @"  \\texttt{barista flow 'java.lang.Object.toString():java.lang.String'}" ;
  @"\\end{center}" ;
  @"It is important to enclose the signature of the method inside quotes," ;
  @"as otherwise the shell would interpret the parentheses."
]

class parameters elements_desc = object

  inherit Command.base_parameters elements_desc as super

  inherit! Command.graph_format as super_graph

  method! switches =
    super#switches @ super_graph#switches

  method! others =
    super#others

end

let make_parameters, switches_of_parameters, others_of_parameters =
  Command.parameters_functions
    (fun () ->
      new parameters
      @"arguments are descriptors of methods to analyze")

let run params =
  let method_signatures = params#get_elements in
  match method_signatures with
  | _ :: _ ->
      let print_header = Utils.make_header_printer UTF8.to_string method_signatures in
      let class_path = params#build_class_path in
      let class_loader = ClassLoader.make_of_class_path class_path in
      List.iter
        (fun class_name ->
          print_header class_name;
          FlowPrinter.print params#get_graph_format class_loader class_name)
        method_signatures;
      ClassPath.close_noerr class_path
  | [] ->
      ()
