(*
 * This file is part of Barista.
 * Copyright (C) 2007-2014 Xavier Clerc.
 *
 * Barista is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Barista is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *)

class type class_definition_mapper = object
  method class_definition : AccessFlag.for_class list ->
    Name.for_class ->
    Name.for_class option ->
    Name.for_class list ->
    Field.t list ->
    Method.t list ->
    Attribute.for_class list ->
    (AccessFlag.for_class list * Name.for_class * Name.for_class option * Name.for_class list * Field.t list * Method.t list * Attribute.for_class list)
  method class_flags : AccessFlag.for_class list -> AccessFlag.for_class list
  method class_name : Name.for_class -> Name.for_class
  method class_extends : Name.for_class option -> Name.for_class option
  method class_implements : Name.for_class list -> Name.for_class list
  method class_fields : Field.t list -> Field.t list
  method class_field : Field.t -> Field.t
  method class_methods : Method.t list -> Method.t list
  method class_method : Method.t -> Method.t
  method class_attributes : Attribute.for_class list -> Attribute.for_class list
  method class_attribute : Attribute.for_class -> Attribute.for_class
  method field_flags : AccessFlag.for_field list -> AccessFlag.for_field list
  method field_name : Name.for_field -> Name.for_field
  method field_descriptor : Descriptor.for_field -> Descriptor.for_field
  method field_attributes : Attribute.for_field list -> Attribute.for_field list
  method field_attribute : Attribute.for_field -> Attribute.for_field
  method regular_method : Method.regular -> Method.regular
  method constructor_method : Method.constructor -> Method.constructor
  method initializer_method : Method.class_initializer -> Method.class_initializer
  method regular_method_flags : AccessFlag.for_method list -> AccessFlag.for_method list
  method regular_method_name : Name.for_method -> Name.for_method
  method regular_method_descriptor : Descriptor.for_method -> Descriptor.for_method
  method regular_method_attributes : Attribute.for_method list -> Attribute.for_method list
  method regular_method_attribute : Attribute.for_method -> Attribute.for_method
  method constructor_flags : AccessFlag.for_constructor list -> AccessFlag.for_constructor list
  method constructor_descriptor : Descriptor.for_parameter list -> Descriptor.for_parameter list
  method constructor_attributes : Attribute.for_method list -> Attribute.for_method list
  method constructor_attribute : Attribute.for_method -> Attribute.for_method
  method initializer_flags : AccessFlag.for_initializer list -> AccessFlag.for_initializer list
  method initializer_attributes : Attribute.for_method list -> Attribute.for_method list
  method initializer_attribute : Attribute.for_method -> Attribute.for_method
end

class default_class_definition_mapper : class_definition_mapper = object (self)

  method class_definition flags name extends implements fields methods attributes =
    let flags = self#class_flags flags in
    let name = self#class_name name in
    let extends = self#class_extends extends in
    let implements = self#class_implements implements in
    let fields = self#class_fields fields in
    let methods = self#class_methods methods in
    let attributes = self#class_attributes attributes in
    (flags, name, extends, implements, fields, methods, attributes)

  method class_flags x = x

  method class_name x = x

  method class_extends x = x

  method class_implements x = x

  method class_fields x = List.map self#class_field x

  method class_field x =
    { Field.flags = self#field_flags x.Field.flags;
      Field.name = self#field_name x.Field.name;
      Field.descriptor = self#field_descriptor x.Field.descriptor;
      Field.attributes = self#field_attributes x.Field.attributes }

  method class_methods x = List.map self#class_method x

  method class_method = function
    | Method.Regular mr -> Method.Regular (self#regular_method mr)
    | Method.Constructor mc -> Method.Constructor (self#constructor_method mc)
    | Method.Initializer mi -> Method.Initializer (self#initializer_method mi)

  method class_attributes x = List.map self#class_attribute x

  method class_attribute x = x

  method field_flags x = x

  method field_name x = x

  method field_descriptor x = x

  method field_attributes x = List.map self#field_attribute x

  method field_attribute x = x

  method regular_method x =
    { Method.flags = self#regular_method_flags x.Method.flags;
      Method.name = self#regular_method_name x.Method.name;
      Method.descriptor = self#regular_method_descriptor x.Method.descriptor;
      Method.attributes = self#regular_method_attributes x.Method.attributes }

  method constructor_method x =
    { Method.cstr_flags = self#constructor_flags x.Method.cstr_flags;
      Method.cstr_descriptor = self#constructor_descriptor x.Method.cstr_descriptor;
      Method.cstr_attributes = self#constructor_attributes x.Method.cstr_attributes }

  method initializer_method x =
    { Method.init_flags = self#initializer_flags x.Method.init_flags;
      Method.init_attributes = self#initializer_attributes x.Method.init_attributes }

  method regular_method_flags x = x

  method regular_method_name x = x

  method regular_method_descriptor x = x

  method regular_method_attributes x = List.map self#regular_method_attribute x

  method regular_method_attribute x = x

  method constructor_flags x = x

  method constructor_descriptor x = x

  method constructor_attributes x = List.map self#constructor_attribute x

  method constructor_attribute x = x

  method initializer_flags x = x

  method initializer_attributes x = List.map self#initializer_attribute x

  method initializer_attribute x = x

end

class type class_definition_iterator = object
  method class_definition : AccessFlag.for_class list ->
    Name.for_class ->
    Name.for_class option ->
    Name.for_class list ->
    Field.t list ->
    Method.t list ->
    Attribute.for_class list ->
    unit
  method class_flags : AccessFlag.for_class list -> unit
  method class_name : Name.for_class -> unit
  method class_extends : Name.for_class option -> unit
  method class_implements : Name.for_class list -> unit
  method class_fields : Field.t list -> unit
  method class_field : Field.t -> unit
  method class_methods : Method.t list -> unit
  method class_method : Method.t -> unit
  method class_attributes : Attribute.for_class list -> unit
  method class_attribute : Attribute.for_class -> unit
  method field_flags : AccessFlag.for_field list -> unit
  method field_name : Name.for_field -> unit
  method field_descriptor : Descriptor.for_field -> unit
  method field_attributes : Attribute.for_field list -> unit
  method field_attribute : Attribute.for_field -> unit
  method regular_method : Method.regular -> unit
  method constructor_method : Method.constructor -> unit
  method initializer_method : Method.class_initializer -> unit
  method regular_method_flags : AccessFlag.for_method list -> unit
  method regular_method_name : Name.for_method -> unit
  method regular_method_descriptor : Descriptor.for_method -> unit
  method regular_method_attributes : Attribute.for_method list -> unit
  method regular_method_attribute : Attribute.for_method -> unit
  method constructor_flags : AccessFlag.for_constructor list -> unit
  method constructor_descriptor : Descriptor.for_parameter list -> unit
  method constructor_attributes : Attribute.for_method list -> unit
  method constructor_attribute : Attribute.for_method -> unit
  method initializer_flags : AccessFlag.for_initializer list -> unit
  method initializer_attributes : Attribute.for_method list -> unit
  method initializer_attribute : Attribute.for_method -> unit
end

class default_class_definition_iterator : class_definition_iterator = object (self)

  method class_definition flags name extends implements fields methods attributes =
    self#class_flags flags;
    self#class_name name;
    self#class_extends extends;
    self#class_implements implements;
    self#class_fields fields;
    self#class_methods methods;
    self#class_attributes attributes

  method class_flags _ = ()

  method class_name _ = ()

  method class_extends _ = ()

  method class_implements _ = ()

  method class_fields x = List.iter self#class_field x

  method class_field x =
    self#field_flags x.Field.flags;
    self#field_name x.Field.name;
    self#field_descriptor x.Field.descriptor;
    self#field_attributes x.Field.attributes

  method class_methods x = List.iter self#class_method x

  method class_method = function
    | Method.Regular mr -> self#regular_method mr
    | Method.Constructor mc -> self#constructor_method mc
    | Method.Initializer mi -> self#initializer_method mi

  method class_attributes x = List.iter self#class_attribute x

  method class_attribute _ = ()

  method field_flags _ = ()

  method field_name _ = ()

  method field_descriptor _ = ()

  method field_attributes x = List.iter self#field_attribute x

  method field_attribute _ = ()

  method regular_method x =
    self#regular_method_flags x.Method.flags;
    self#regular_method_name x.Method.name;
    self#regular_method_descriptor x.Method.descriptor;
    self#regular_method_attributes x.Method.attributes

  method constructor_method x =
    self#constructor_flags x.Method.cstr_flags;
    self#constructor_descriptor x.Method.cstr_descriptor;
    self#constructor_attributes x.Method.cstr_attributes

  method initializer_method x =
    self#initializer_flags x.Method.init_flags;
    self#initializer_attributes x.Method.init_attributes

  method regular_method_flags _ = ()

  method regular_method_name _ = ()

  method regular_method_descriptor _ = ()

  method regular_method_attributes x = List.iter self#regular_method_attribute x

  method regular_method_attribute _ = ()

  method constructor_flags _ = ()

  method constructor_descriptor _ = ()

  method constructor_attributes x = List.iter self#constructor_attribute x

  method constructor_attribute _ = ()

  method initializer_flags _ = ()

  method initializer_attributes x = List.iter self#initializer_attribute x

  method initializer_attribute _ = ()

end

class type class_definition_folder = object ('self)
  method class_definition : AccessFlag.for_class list ->
    Name.for_class ->
    Name.for_class option ->
    Name.for_class list ->
    Field.t list ->
    Method.t list ->
    Attribute.for_class list ->
    'self
  method class_flags : AccessFlag.for_class list -> 'self
  method class_name : Name.for_class -> 'self
  method class_extends : Name.for_class option -> 'self
  method class_implements : Name.for_class list -> 'self
  method class_fields : Field.t list -> 'self
  method class_field : Field.t -> 'self
  method class_methods : Method.t list -> 'self
  method class_method : Method.t -> 'self
  method class_attributes : Attribute.for_class list -> 'self
  method class_attribute : Attribute.for_class -> 'self
  method field_flags : AccessFlag.for_field list -> 'self
  method field_name : Name.for_field -> 'self
  method field_descriptor : Descriptor.for_field -> 'self
  method field_attributes : Attribute.for_field list -> 'self
  method field_attribute : Attribute.for_field -> 'self
  method regular_method : Method.regular -> 'self
  method constructor_method : Method.constructor -> 'self
  method initializer_method : Method.class_initializer -> 'self
  method regular_method_flags : AccessFlag.for_method list -> 'self
  method regular_method_name : Name.for_method -> 'self
  method regular_method_descriptor : Descriptor.for_method -> 'self
  method regular_method_attributes : Attribute.for_method list -> 'self
  method regular_method_attribute : Attribute.for_method -> 'self
  method constructor_flags : AccessFlag.for_constructor list -> 'self
  method constructor_descriptor : Descriptor.for_parameter list -> 'self
  method constructor_attributes : Attribute.for_method list -> 'self
  method constructor_attribute : Attribute.for_method -> 'self
  method initializer_flags : AccessFlag.for_initializer list -> 'self
  method initializer_attributes : Attribute.for_method list -> 'self
  method initializer_attribute : Attribute.for_method -> 'self
end

class default_class_definition_folder : class_definition_folder = object (self)

  method class_definition flags name extends implements fields methods attributes =
    let self = self#class_flags flags in
    let self = self#class_name name in
    let self = self#class_extends extends in
    let self = self#class_implements implements in
    let self = self#class_fields fields in
    let self = self#class_methods methods in
    let self = self#class_attributes attributes in
    self

  method class_flags _ = self

  method class_name _ = self

  method class_extends _ = self

  method class_implements _ = self

  method class_fields x = List.fold_left (fun acc elem -> acc#class_field elem) self x

  method class_field x =
    let self = self#field_flags x.Field.flags in
    let self = self#field_name x.Field.name in
    let self = self#field_descriptor x.Field.descriptor in
    let self = self#field_attributes x.Field.attributes in
    self

  method class_methods x = List.fold_left (fun acc elem -> acc#class_method elem) self x

  method class_method = function
    | Method.Regular mr -> self#regular_method mr
    | Method.Constructor mc -> self#constructor_method mc
    | Method.Initializer mi -> self#initializer_method mi

  method class_attributes x = List.fold_left (fun acc elem -> acc#class_attribute elem) self x

  method class_attribute _ = self

  method field_flags _ = self

  method field_name _ = self

  method field_descriptor _ = self

  method field_attributes x = List.fold_left (fun acc elem -> acc#field_attribute elem) self x

  method field_attribute _ = self

  method regular_method x =
    let self = self#regular_method_flags x.Method.flags in
    let self = self#regular_method_name x.Method.name in
    let self = self#regular_method_descriptor x.Method.descriptor in
    let self = self#regular_method_attributes x.Method.attributes in
    self

  method constructor_method x =
    let self = self#constructor_flags x.Method.cstr_flags in
    let self = self#constructor_descriptor x.Method.cstr_descriptor in
    let self = self#constructor_attributes x.Method.cstr_attributes in
    self

  method initializer_method x =
    let self = self#initializer_flags x.Method.init_flags in
    let self = self#initializer_attributes x.Method.init_attributes in
    self

  method regular_method_flags _ = self

  method regular_method_name _ = self

  method regular_method_descriptor _ = self

  method regular_method_attributes x = List.fold_left (fun acc elem -> acc#regular_method_attribute elem) self x

  method regular_method_attribute _ = self

  method constructor_flags _ = self

  method constructor_descriptor _ = self

  method constructor_attributes x = List.fold_left (fun acc elem -> acc#constructor_attribute elem) self x

  method constructor_attribute _ = self

  method initializer_flags _ = self

  method initializer_attributes x = List.fold_left (fun acc elem -> acc#initializer_attribute elem) self x

  method initializer_attribute _ = self

end

