(*
 * This file is part of Barista.
 * Copyright (C) 2007-2014 Xavier Clerc.
 *
 * Barista is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Barista is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *)

BARISTA_ERROR =
  | Method_has_no_code -> "method has no code"

let print_to_buffer buffer fmt ld s =
  let meth =
    if UTF8.contains @':' s then
      let tmp = Lookup.for_regular_method false ~open_packages:[ @"java.lang" ] ld s in
      Method.Regular (snd tmp.Lookup.value)
    else
      let tmp = Lookup.for_constructor false ~open_packages:[ @"java.lang" ] ld s in
      Method.Constructor (snd tmp.Lookup.value) in
  let code =
    try
      Attribute.extract_code
        ((match meth with
        | Method.Regular { Method.attributes = a; _ } -> a
        | Method.Constructor { Method.cstr_attributes = a; _ } -> a
        | Method.Initializer { Method.init_attributes = a; _ } -> a) :> Attribute.t list)
    with _ -> fail Method_has_no_code in
  let graph =
    ControlFlow.graph_of_instructions
      code.Attribute.code
      code.Attribute.exception_table in
  let graph =
    ControlFlow.to_plain_graph
      (fun (ofs, _) -> UTF8.of_string (Printf.sprintf "[offset %ld]" ofs))
      (fun _ -> @"")
      graph in
  Graph.dump fmt graph buffer

let print_to_stream chan fmt ld s =
  let buffer = UTF8Buffer.make () in
  print_to_buffer buffer fmt ld s;
  let writer = UTF8LineWriter.make_of_stream chan in
  UTF8LineWriter.write_line
    writer
    (UTF8Buffer.contents buffer);
  UTF8LineWriter.flush writer

let print fmt ld s =
  let s =
    if (UTF8.length s = 0) || (UTF8.contains @':' s) then
      s
    else
      UTF8.(++) s @":_" in
  print_to_stream OutputStream.stdout fmt ld s
