(*
 * This file is part of Barista.
 * Copyright (C) 2007-2014 Xavier Clerc.
 *
 * Barista is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Barista is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *)

(** Access to archives (zip, jar, etc.) in read mode.

    This definition is independent of the actual implementation, that can
    be based on either the OCaml standard library, or the OCaml-Java
    runtime library. *)


type t
(** The type of archives. *)

BARISTA_ERROR =
  | Unable_to_open_archive of Path.t
  | Unknown_entry of UTF8.t
  | Unable_to_read_entry of UTF8.t
  | Unable_to_close_archive

val make_of_path : Path.t -> t
(** Opens the archive whose path is passed.

    Raises [Exception] if the archive cannot be opened. *)

val get_comment : t -> UTF8.t
(** Returns the comment associated with the passed archive. *)

val find_entry : t -> UTF8.t -> ArchiveEntry.t
(** [find_entry archive name] looks for an entry named [name] in
    [archive], raising [Exception] if such an entry does not exist. *)

val iter_entries : (ArchiveEntry.t -> unit) -> t -> unit
(** Iteration over archive entries. The order of entries is unspecified. *)

val fold_entries : ('a -> ArchiveEntry.t -> 'a) -> 'a -> t -> 'a
(** Fold over archive entries. The order of entries is unspecified. *)

val stream_of_entry : t -> ArchiveEntry.t -> InputStream.t
(** Returns a stream allowing to read the data of the passed entry.

    Raises [Exception] if data cannot be read. *)

val bytes_of_entry : t -> ArchiveEntry.t -> Bytes.t
(** Returns the data of the passed entry.

    Raises [Exception] if data cannot be read. *)

val close : t -> unit
(** Closes the passed archive.

    Raises [Exception] if an error occurs while closing the archive. *)

val close_noerr : t -> unit
(** Same as [close] but raised exceptions are silently discarded. *)
