(*
 * This file is part of Barista.
 * Copyright (C) 2007-2014 Xavier Clerc.
 *
 * Barista is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Barista is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *)

(** Function allowing to search for classes, fields, constructors, and
  methods using UTF8 queries. *)


(** {6 Base types and utility functions} *)

BARISTA_ERROR =
  | Invalid_class_query of UTF8.t
  | Cannot_find_class of UTF8.t
  | Generic_arity_mismatch of UTF8.t * int * int
  | Generic_cannot_be_parametrized
  | Primitive_cannot_be_generic_argument
  | Wildcard_cannot_be_generic_argument
  | Varargs_cannot_be_generic_argument
  | Generic_variable_bound_to_different_values of UTF8.t
  | Void_cannot_be_an_array_element_type
  | Invalid_generic_bound
  | Ambiguity_in_generic_bound
  | No_class_match of UTF8.t
  | Several_class_matches of Name.for_class list
  | Invalid_field_query of UTF8.t
  | No_field_match of UTF8.t * Name.for_field
  | Several_field_matches of (Name.for_class * Name.for_field) list
  | Invalid_constructor_query of UTF8.t
  | No_constructor_match of UTF8.t
  | Several_constructor_matches of (Name.for_class * Descriptor.for_parameter list) list
  | Invalid_regular_method_query of UTF8.t
  | No_regular_method_match of UTF8.t * Name.for_method
  | Several_regular_method_matches of (Name.for_class * Name.for_method * Descriptor.for_method) list
  | Cannot_use_wide_wildcard_for_field_type
  | Wide_wildcard_cannot_be_combined_with_another_parameter_type
  | Varargs_can_only_appear_at_the_end_of_a_type

type java_type =
  [ `Boolean
  | `Byte
  | `Char
  | `Double
  | `Float
  | `Int
  | `Long
  | `Short
  | `Void
  | `Class of ClassDefinition.t
  | `Array of 'a
  | `Varargs of 'a
  | `Generic of UTF8.t
  | `Wildcard of bool ] constraint 'a = non_void_java_type
(** Definition of Java types, akin to [Descriptor.java_type] but with
    extra cases:
    - [`Generic id] to encode a generic type variable designed by an
      identifier;
    - [`Wildcard false] to encode the special name ["_"] that can be used
      to match any type, and [`Wildcard true] that can be used to match
      any number of types. *)
and non_void_java_type =
  [ `Boolean
  | `Byte
  | `Char
  | `Double
  | `Float
  | `Int
  | `Long
  | `Short
  | `Class of ClassDefinition.t
  | `Array of 'a
  | `Varargs of 'a
  | `Generic of UTF8.t
  | `Wildcard of bool ] constraint 'a = non_void_java_type
(** Definition of Java types, akin to [Descriptor.non_void_java_type] but
    with extra cases:
    - [`Generic id] to encode a generic type variable designed by an
      identifier;
    - [`Wildcard false] to encode the special name ["_"] that can be used
      to match any type, and [`Wildcard true] that can be used to match
      any number of types. *)

type substitution = java_type UTF8.Map.t
(** The type of substitutions, mapping identifiers of formal generic type
    parameters to their effective types. *)

type 'a result = {
    value : 'a; (** The actual result of the query. *)
    with_generics : bool; (** Whether the query was generic-aware. *)
    substitution : substitution; (** The substitution of generic type parameters. *)
  }
(** The type of query results. *)

val equal_java_type : java_type -> java_type -> bool
(** Equality over Java types. *)

val matches_descriptor : substitution -> java_type -> Descriptor.java_type -> bool
(** Tests whether a Java type matches a given Java type descriptor.

    The passed substitution is used to replace generic type variables. *)


(** {6 Support for classes} *)

type for_class
(** The type of queries about classes. *)

val utf8_for_class : for_class -> UTF8.t
(** Returns the string representation of the passed query. *)

val make_for_class_from_utf8 : bool -> UTF8.t -> for_class
(** [make_for_class_from_utf8 generics s] returns a query looking for
    classes according to string [s], [generics] indicating whether
    generic information should be used. A query should be a class name
    (possibly unqualified), possibly followed by generic parameters
    between angle brackets (if [generics] is [true]).

    Raises [Exception] if [s] is not well-formed. *)

val search_for_classes : ?open_packages:UTF8.t list -> ClassLoader.t -> for_class -> ClassDefinition.t result list
(** [search_for_classes ~open_packages:op cl q] returns a list of classes
    matching query [q] found in class loader [cl], [op] giving the list
    of package prefixes that need not to appear (thus allowing to use
    short rather than fully-qualified class names).

    Query examples:
    - ["String"];
    - ["java.util.List"];
    - ["java.util.List<T>"];
    - ["java.util.List<int[]>"];
    - ["java.util.List<Object>"].

    The returned substitutions are:
    - empty if no generic parameter appears i the query;
    - binding formal generics parameters ["Fi"] appearing in the class
      definition to effective parameters ["Ei"] appearing in the query.

    Some examples, based on the {i java.util.List<E>} class:
    - query ["java.util.List"] will result in an empty substitution;
    - query ["java.util.List<T>"] will bind ["E"] to [`Generic "T"];
    - query ["java.util.List<java.lang.Integer>"] will bind ["E"] to
      [`Class cd] where cd is the class definition for
      {i java.lang.Integer}. *)

val search_for_class : ?open_packages:UTF8.t list -> ClassLoader.t -> for_class -> ClassDefinition.t result
(** Akin to [search_for_classes], except that [Exception] is raised if
    the returned list contains zero or more than one class. *)

val for_classes : bool -> ?open_packages:UTF8.t list -> ClassLoader.t -> UTF8.t -> ClassDefinition.t result list
(** [for_classes generics ~open_packages:op cl s] is a bare shorthand for
    [search_for_classes ~open_packages:op cl
    (make_for_class_from_utf8 generics s)]. *)

val for_class : bool -> ?open_packages:UTF8.t list -> ClassLoader.t -> UTF8.t -> ClassDefinition.t result
(** [for_class generics ~open_packages:op cl s] is a bare shorthand for
    [search_for_class ~open_packages:op cl
    (make_for_class_from_utf8 generics s)]. *)


(** {6 Support for fields} *)

type for_field
(** The type of queries about fields. *)

val utf8_for_field : for_field -> UTF8.t
(** Returns the string representation of the passed query. *)

val make_for_field_from_utf8 : bool -> UTF8.t -> for_field
(** [make_for_field_from_utf8 generics s] returns a query looking for
    fields according to string [s], [generics] indicating whether
    generic information should be used. A query should successively
    contain:
    - a class name (possibly unqualified), possibly followed by generic
      parameters between angle brackets (if [generics] is [true]);
    - a dot character;
    - a field name;
    - a colon character;
    - a type descriptor (or ["_"] to denote any type).

    Raises [Exception] if [s] is not well-formed. *)

val search_for_fields : ?open_packages:UTF8.t list -> ClassLoader.t -> for_field -> (ClassDefinition.t * Field.t) result list
(** [search_for_fields ~open_packages:op cl q] returns a list of
    class/field couples matching query [q] found in class loader [cl],
    [op] giving the list of package prefixes that need not to appear
    (thus allowing to use short rather than fully-qualified class names).

    Query examples:
    - ["Integer.MIN_VALUE:_"];
    - ["Integer.digits:char[]"];
    - ["java.util.LinkedList.size:int"];
    - ["java.util.LinkedList<T>.size:int"];
    - ["java.util.LinkedList<Object>.size:int"]. *)

val search_for_field : ?open_packages:UTF8.t list -> ClassLoader.t -> for_field -> (ClassDefinition.t * Field.t) result
(** Akin to [search_for_fields], except that [Exception] is raised if the
    returned list contains zero or more than one field. *)

val for_fields : bool -> ?open_packages:UTF8.t list -> ClassLoader.t -> UTF8.t -> (ClassDefinition.t * Field.t) result list
(** [for_fields generics ~open_packages:op cl s] is a bare shorthand for
    [search_for_fields ~open_packages:op cl
    (make_for_field_from_utf8 generics s)]. *)

val for_field : bool -> ?open_packages:UTF8.t list -> ClassLoader.t -> UTF8.t -> (ClassDefinition.t * Field.t) result
(** [for_field generics ~open_packages:op cl s] is a bare shorthand for
    [search_for_field ~open_packages:op cl
    (make_for_field_from_utf8 generics s)]. *)


(** {6 Support for constructors} *)

type for_constructor
(** The type of queries about constructors. *)

val utf8_for_constructor : for_constructor -> UTF8.t
(** Returns the string representation of the passed query. *)

val make_for_constructor_from_utf8 : bool -> UTF8.t -> for_constructor
(** [make_for_constructor_from_utf8 generics s] returns a query looking
    for constructors according to string [s], [generics] indicating
    whether generic information should be used. A query should
    successively contain:
    - a class name (possibly unqualified), possibly followed by generic
      parameters between angle brackets (if [generics] is [true]);
    - an opening-parenthesis character;
    - a list of comma-separated type descriptors;
    - a closing-parenthesis character.
    Any parameter type descriptor can be remplaced by ["_"] to denote any
    type.

    Raises [Exception] if [s] is not well-formed. *)

val search_for_constructors : ?open_packages:UTF8.t list -> ClassLoader.t -> for_constructor -> (ClassDefinition.t * Method.constructor) result list
(** [search_for_constructors ~open_packages:op cl q] returns a list of
    class/constructor couples matching query [q] found in class loader
    [cl], [op] giving the list of package prefixes that need not to appear
    (thus allowing to use short rather than fully-qualified class names).

    Query examples:
    - ["Object()"];
    - ["Integer(int)"];
    - ["java.util.LinkedList()"];
    - ["java.util.LinkedList<T>()"];
    - ["java.util.LinkedList<Object>()"]. *)

val search_for_constructor : ?open_packages:UTF8.t list -> ClassLoader.t -> for_constructor -> (ClassDefinition.t * Method.constructor) result
(** Akin to [search_for_constructors], except that [Exception] is raised
    if the returned list contains zero or more than one constructor. *)

val for_constructors : bool -> ?open_packages:UTF8.t list -> ClassLoader.t -> UTF8.t -> (ClassDefinition.t * Method.constructor) result list
(** [for_constructors generics ~open_packages:op cl s] is a bare shorthand
    for [search_for_constructors ~open_packages:op cl
    (make_for_constructor_from_utf8 generics s)]. *)

val for_constructor : bool -> ?open_packages:UTF8.t list -> ClassLoader.t -> UTF8.t -> (ClassDefinition.t * Method.constructor) result
(** [for_constructor generics ~open_packages:op cl s] is a bare shorthand
    for [search_for_constructor ~open_packages:op cl
    (make_for_constructor_from_utf8 generics s)]. *)


(** {6 Support for regular methods} *)

type for_regular_method
(** The type of queries about regular methods. *)

val utf8_for_regular_method : for_regular_method -> UTF8.t
(** Returns the string representation of the passed query. *)

val make_for_regular_method_from_utf8 : bool -> UTF8.t -> for_regular_method
(** [make_for_regular_method_from_utf8 generics s] returns a query looking
    for regular methods according to string [s], [generics] indicating
    whether generic information should be used. A query should
    successively contain:
    - a class name (possibly unqualified), possibly followed by generic
      parameters between angle brackets (if [generics] is [true]);
    - a dot character;
    - a method name;
    - an opening-parenthesis character;
    - a list of comma-separated type descriptors;
    - a closing-parenthesis character;
    - a colon character;
    - a type descriptor.
    Any parameter type descriptor can be remplaced by ["_"] to denote any
    type.

    Raises [Exception] if [s] is not well-formed. *)

val search_for_regular_methods : ?open_packages:UTF8.t list -> ClassLoader.t -> for_regular_method -> (ClassDefinition.t * Method.regular) result list
(** [search_for_regular_methods ~open_packages:op cl q] returns a list of
    class/method couples matching query [q] found in class loader [cl],
    [op] giving the list of package prefixes that need not to appear
    (thus allowing to use short rather than fully-qualified class names).

    Query examples:
    - ["Object.clone()"];
    - ["Object.equals(Object)"];
    - ["java.util.LinkedList.get(int):Object"];
    - ["java.util.LinkedList<T>.get(int):T"];
    - ["java.util.LinkedList<Object>.get(int):Object"]. *)

val search_for_regular_method : ?open_packages:UTF8.t list -> ClassLoader.t -> for_regular_method -> (ClassDefinition.t * Method.regular) result
(** Akin to [search_for_regular_methods], except that [Exception] is raised
    if the returned list contains zero or more than one regular method. *)

val for_regular_methods : bool -> ?open_packages:UTF8.t list -> ClassLoader.t -> UTF8.t -> (ClassDefinition.t * Method.regular) result list
(** [for_regular_methods generics ~open_packages:op cl s] is a bare shorthand
    for [search_for_regular_methods ~open_packages:op cl
    (make_for_regular_method_from_utf8 generics s)]. *)

val for_regular_method : bool -> ?open_packages:UTF8.t list -> ClassLoader.t -> UTF8.t -> (ClassDefinition.t * Method.regular) result
(** [for_regular_method generics ~open_packages:op cl s] is a bare shorthand
    for [search_for_regular_method ~open_packages:op cl
    (make_for_regular_method_from_utf8 generics s)]. *)

