(*
 * This file is part of Barista.
 * Copyright (C) 2007-2014 Xavier Clerc.
 *
 * Barista is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Barista is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *)

(** Utility functions to easily build Java archives. *)

BARISTA_ERROR =
  | Duplicate_entry of UTF8.t
  | Duplicate_service of UTF8.t

type t
(** The type of archive builders. *)

val make : ?merge_services:bool -> ?fail_on_duplicate:bool -> ?generate_index:bool -> Path.t -> t
(* Constructs an archive builder based on the file whose path is passed,
   and whose behaviour is defined by optional parameters:
   - [merge_services] (defaulting to [false]) indicates whether entries
     for services should be merged;
   - [fail_on_duplicate] (defaulting to [false]) indicates whether an
     exception should be raised when two entries have the same path
   inside the archive;
   - [generate_index] (defaulting to [false]) indicates whether an index
     should be generated at archive closing.

   Raises an exception if archive cannot be created. *)

val add_entry : t -> ?prefix:UTF8.t -> UTF8.t -> Bytes.t -> unit
(** [add_entry builder ~prefix name bytes] adds an entry with path [name]
    prefixed by [prefix] (defaulting to [""]) and data [bytes] to archive
    builder [builder].

    Raises [Error] if [builder] already contains an entry with the same
    path, and [builder] was passed [true] for [fail_on_duplicate] at
    creation. *)

val add_entry_from_file : t -> ?prefix:UTF8.t -> UTF8.t -> Path.t -> unit
(** [add_entry_from_file builder ~prefix name path] adds an entry with
    path [name] prefixed by [prefix] (defaulting to [""]) and contents
    read from file [path] to archive builder [builder].

    Raises [Error] if [builder] already contains an entry with the same
    path, and [builder] was passed [true] for [fail_on_duplicate] at
    creation. *)

val add_entry_from_class : t -> ?prefix:UTF8.t -> ClassDefinition.t -> unit
(** [add_entry_from_class builder ~prefix def] adds an entry with path
    and contents based on the information from class definition [def],
    prefixing class name by by [prefix] (defaulting to [""]), to archive
    builder [builder].

    Raises [Error] if [builder] already contains an entry with the same
    path, and [builder] was passed [true] for [fail_on_duplicate] at
    creation. *)

val add_entries_from_archive : t -> ?prefix:UTF8.t -> ArchiveFile.t -> unit
(** [add_entries_from_archive builder ~prefix arch] adds to archive
    builder [builder] all the entries from the archive [arch], prefixing
    paths with [prefix] (defaulting to [""]).

    Raises [Error] if [builder] already contains an entry with the same
    path, and [builder] was passed [true] for [fail_on_duplicate] at
    creation. *)

val add_entries_from_archive_file : t -> ?prefix:UTF8.t -> Path.t -> unit
(** [add_entries_from_archive_file builder ~prefix path] adds to archive
    builder [builder] all the entries from the archive whose name is
    [path], prefixing entry paths with [prefix] (defaulting to [""]).

    Raises [Error] if [builder] already contains an entry with the same
    path, and [builder] was passed [true] for [fail_on_duplicate] at
    creation. *)

val add_entries_from_archive_files : t -> ?prefix:UTF8.t -> Path.t list -> unit
(** [add_entries_from_archive_files builder ~prefix paths] adds to
    archive builder [builder] all the entries from thes archives whose
    names are [paths], prefixing entry paths with [prefix] (defaulting to
    [""]).

    Raises [Error] if [builder] already contains an entry with the same
    path, and [builder] was passed [true] for [fail_on_duplicate] at
    creation. *)

val add_manifest : t -> Manifest.t -> unit
(** [add_manifest builder manifest] adds an entry based on the passed
    manifest [manifest] to archive builder [builder].

    Raises [Error] if [builder] already contains an entry with the same
    path, and [builder] was passed [true] for [fail_on_duplicate] at
    creation. *)

val close : t -> unit
(** Closes the passed archive builder.

    Raises [Error] if index generation was requested, and results in a
    duplicate entry. *)

val close_noerr : t -> unit
(** Same as [close] but raised exceptions are silently discarded. *)
