(*
 * This file is part of Barista.
 * Copyright (C) 2007-2014 Xavier Clerc.
 *
 * Barista is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Barista is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *)


(* Graph representation *)

type vertex = {
    vertex_id : UTF8.t;
    vertex_cluster : UTF8.t option;
    vertex_label : UTF8.t option;
    vertex_properties : UTF8.t UTF8.Map.t;
  }
and edge = {
    edge_id : UTF8.t;
    edge_directed : bool;
    edge_vertices : (UTF8.t * UTF8.t) list;
    edge_label : UTF8.t option;
    edge_properties : UTF8.t UTF8.Map.t;
  }
and t = {
    vertices : vertex list;
    edges : edge list;
    vertice_types : UTF8.t UTF8.Map.t;
    edge_types : UTF8.t UTF8.Map.t;
  }


(* Graph builders *)

type builder = {
    mutable curr_vertices : vertex list;
    mutable curr_edges : edge list;
    mutable curr_vertice_types : UTF8.t UTF8.Map.t;
    mutable curr_edge_types : UTF8.t UTF8.Map.t;
  }
(** The type of graph builder. *)

let make () =
  { curr_vertices = [];
    curr_edges = [];
    curr_vertice_types = UTF8.Map.empty;
    curr_edge_types = UTF8.Map.empty }

let to_graph b =
  { vertices = List.rev b.curr_vertices;
    edges = List.rev b.curr_edges;
    vertice_types = b.curr_vertice_types;
    edge_types = b.curr_edge_types }

let none = @"none"

let map_of_list l =
  List.fold_right
    (fun (key, value) acc ->
      UTF8.Map.add key value acc)
    l
    UTF8.Map.empty

let add_vertex ?(cluster = none) ?(label = none) ?(properties = []) ~id b =
  let v =
    { vertex_id = id;
      vertex_cluster = (if cluster == none then None else Some cluster);
      vertex_label = (if label == none then None else Some label);
      vertex_properties = map_of_list properties } in
  b.curr_vertices <- v :: b.curr_vertices

let add_edge ?(directed = false) ?(label = none) ?(properties = []) ~id ~vertices b =
  let e =
    { edge_id = id;
      edge_directed = directed;
      edge_vertices = vertices;
      edge_label = (if label == none then None else Some label);
      edge_properties = map_of_list properties } in
  b.curr_edges <- e :: b.curr_edges

let add_vertice_type n t b =
  b.curr_vertice_types <- UTF8.Map.add n t b.curr_vertice_types

let add_edge_type n t b =
  b.curr_edge_types <- UTF8.Map.add n t b.curr_edge_types


(* Output *)

type format =
  | Dot
  | GraphML
  | GEXF

let all_formats = [
  Dot ;
  GraphML ;
  GEXF ;
]

let string_of_format = function
  | Dot -> "Dot"
  | GraphML -> "GraphML"
  | GEXF -> "GEXF"

let utf8_of_format = function
  | Dot -> @"Dot"
  | GraphML -> @"GraphML"
  | GEXF -> @"GEXF"

let identifier_of_format = function
  | Dot -> @"dot"
  | GraphML -> @"graphml"
  | GEXF -> @"gexf"

(* transforms hyperedges into simple edges,
   returning <edge list, additional vertices> *)
let simplify_edges l l' =
  let ids = ref UTF8.Set.empty in
  let next_id = ref 0 in
  let rec get_id s =
    let res = SPRINTF ("hyper_%s_%d" s !next_id) in
    incr next_id;
    if UTF8.Set.mem res !ids then begin
      get_id s
    end else begin
      ids := UTF8.Set.add res !ids;
      res
    end in
  List.iter (fun e -> ids := UTF8.Set.add e.edge_id !ids) l;
  List.iter (fun v -> ids := UTF8.Set.add v.vertex_id !ids) l';
  let l =
    List.map
      (fun e ->
        if (List.length e.edge_vertices) <= 2 then
          [e], []
        else begin
          let additional_vertex_id = get_id @"node" in
          let additional_vertex =
            { vertex_id = additional_vertex_id;
              vertex_cluster = None;
              vertex_label = e.edge_label;
              vertex_properties = UTF8.Map.empty } in
          let src_edge =
            { edge_id = get_id @"vertex";
              edge_directed = false;
              edge_vertices = [List.hd e.edge_vertices;
                               additional_vertex_id, @""];
              edge_label = None;
              edge_properties = e.edge_properties } in
          let edges =
            List.map
              (fun (dst, lbl) ->
                 { edge_id = get_id @"edge";
                   edge_directed = e.edge_directed;
                   edge_vertices = [additional_vertex_id, @""; dst, lbl];
                   edge_label = None;
                   edge_properties = UTF8.Map.empty })
              (List.tl e.edge_vertices) in
          src_edge :: edges,
          [additional_vertex]
        end)
      l in
  let l, l' = List.split l in
  (List.flatten l), (List.flatten l')

let dump_dot g buff =
  let add_endline x = UTF8Buffer.add_endline buff x in
  let make_ident s =
    let len = UTF8.length s in
    let buff = UTF8Buffer.make_of_size (2 * len) in
    for i = 0 to pred len do
      let ch = UTF8.get s i in
      if UChar.is_identifier_part ch  then
        UTF8Buffer.add_char buff ch
      else
        UTF8Buffer.add_char buff @'_'
    done;
    UTF8Buffer.contents buff in
  let idents = ref UTF8.Map.empty in
  let rec free_ident suffix base =
    let x =
      if suffix < 0 then
        base
      else
        SPRINTF ("%s%d" base suffix) in
    let present = try ignore (UTF8.Map.find x !idents); true with Not_found -> false in
    if present then free_ident (succ suffix) base else x in
  let ident s =
    try
      UTF8.Map.find s !idents
    with Not_found ->
      free_ident (-1) (make_ident s) in
  let make_label lbl props =
    let conv (n, v) = SPRINTF ("%s: %s" n v) in
    let prefix = match lbl with Some s -> s | None -> @"-" in
    let suffix =
      props
      |> UTF8.Map.bindings
      |> List.sort (fun (n1, _) (n2, _) -> UTF8.compare n1 n2)
      |> UTF8.concat_sep_map @"<br/>" conv in
    let res =
      if (UTF8.length suffix) = 0 then
        prefix
      else
        SPRINTF ("%s<br/>%s" prefix suffix) in
    let len = UTF8.length res in
    let buff = UTF8Buffer.make_of_size (2 * len) in
    for i = 0 to pred len do
      match UTF8.get res i with
      | @'\n' ->
          UTF8Buffer.add_string buff @"<br/>"
      | @'\"' ->
          UTF8Buffer.add_string buff @"&quot;"
      | ch ->
          UTF8Buffer.add_char buff ch
    done;
    UTF8Buffer.contents buff in
  let edges, additional_vertices = simplify_edges g.edges g.vertices in
  let vertices = g.vertices @ additional_vertices in
  add_endline @"digraph {";
  List.iter
    (fun v ->
      BPRINTF ("  %s [shape=box,label=<%s>]\n"
               buff
               (ident v.vertex_id)
               (make_label v.vertex_label v.vertex_properties)))
    vertices;
  let clusters =
    List.fold_left
      (fun clusters vertex ->
        match vertex.vertex_cluster with
        | Some cluster when (UTF8.length cluster) > 0 ->
            let old =
              try
                UTF8.Map.find cluster clusters
              with Not_found -> [] in
            (UTF8.Map.add cluster (vertex :: old) clusters)
        | _ ->
            clusters)
      UTF8.Map.empty
      vertices in
  UTF8.Map.iter
    (fun cluster vertices ->
      BPRINTF ("subgraph %s {" buff (ident cluster));
      List.iter
        (fun v ->
          BPRINTF ("  %s;" buff (ident v.vertex_id)))
        vertices;
      add_endline @" }")
    clusters;
  List.iter
    (fun e ->
      match e.edge_vertices with
      | x :: y :: [] ->
          BPRINTF ("  %s -> %s [label=<%s>]\n" buff
                     (ident (fst x))
                     (ident (fst y))
                     (make_label e.edge_label e.edge_properties))
      | _ -> ())
    edges;
  add_endline @"}"

let dump_graphml g buff =
  let get_data m =
    UTF8.Map.fold
      (fun n v acc ->
        let tag = @"data" in
        let attrs = [ @"key", n ] in
        let text = XML.pcdata v in
        (XML.tag ~tag ~attrs ~children:[text]) :: acc)
      m
      [] in
  let get_nodes l =
    List.map
      (fun v ->
        let tag = @"node" in
        let attrs = [ @"id", v.vertex_id ] in
        let children = get_data v.vertex_properties in
        XML.tag ~tag ~attrs ~children)
      l in
  let get_keys cls types =
    UTF8.Map.fold
      (fun n t acc ->
        let tag = @"key" in
        let attrs = [
          @"id", n ;
          @"for", cls ;
          @"attr.name", n ;
          @"attr.type", t ;
        ] in
        let children = [] in
        (XML.tag ~tag ~attrs ~children) :: acc)
      types
      [] in

  let clusters, outer =
    List.fold_left
      (fun (clusters, outer) vertex ->
        match vertex.vertex_cluster with
        | Some cluster when (UTF8.length cluster) > 0 ->
            let old =
              try
                UTF8.Map.find cluster clusters
              with Not_found -> [] in
            (UTF8.Map.add cluster (vertex :: old) clusters, outer)
        | _ ->
            (clusters, vertex :: outer))
      (UTF8.Map.empty, [])
      g.vertices in
  let clusters =
    UTF8.Map.fold
      (fun cluster vertices acc ->
        let tag = @"node" in
        let attrs = [ @"id", cluster ] in
        let child = XML.tag ~tag:@"graph" ~attrs:[] ~children:(get_nodes vertices) in
        XML.tag ~tag ~attrs ~children:[child] :: acc)
      clusters
      [] in

  let edges =
    List.map
      (fun e ->
        match e.edge_vertices with
        | x :: y :: [] ->
            let tag = @"edge" in
            let attrs = [ @"id", e.edge_id ;
                          @"directed", (if e.edge_directed then @"true" else @"false") ;
                          @"source", (fst x) ;
                          @"target", (fst y) ] in
            let children = get_data e.edge_properties in
            XML.tag ~tag ~attrs ~children
        | _ ->
            let tag = @"hyperedge" in
            let attrs = [] in
            let children =
              (List.map
                 (fun (v, _) ->
                   let tag = @"endpoint" in
                   let attrs = [ @"node", v ] in
                   let children = [] in
                   XML.tag ~tag ~attrs ~children)
                 e.edge_vertices)
              @ (get_data e.edge_properties) in
            XML.tag ~tag ~attrs ~children)
      g.edges in

  let graph =
    let tag = @"graph" in
    let attrs = [ @"id", @"Java_classes"; @"edgedefault", @"directed" ] in
    let children = (get_nodes outer) @ clusters @ edges in
    XML.tag ~tag ~attrs ~children in

  let document =
    let tag = @"graphml" in
    let attrs = [ @"xmlns", @"http://graphml.graphdrawing.org/xmlns" ;
                  @"xmlns:xsi", @"http://www.w3.org/2001/XMLSchema-instance" ;
                  @"xsi:schemaLocation", @"http://graphml.graphdrawing.org/xmlns http://graphml.graphdrawing.org/xmlns/1.0/graphml.xsd" ] in
    let children =
      (get_keys @"node" g.vertice_types)
      @ (get_keys @"edge" g.edge_types)
      @ [ graph ] in
    XML.tag ~tag ~attrs ~children in
  XML.dump document buff

let dump_gexf g buff =
  let timestamp = UTF8.of_string (OS.timestamp ()) in
  let get_attributes cls types =
    let tag = @"attributes" in
    let attrs = [ @"class", cls ] in
    let children =
      UTF8.Map.fold
        (fun n t acc ->
          let tag = @"attribute" in
          let attrs = [ @"id", n; @"title", n; @"type", t ] in
          let children = [] in
          (XML.tag ~tag ~attrs ~children) :: acc)
        types
        [] in
    XML.tag ~tag ~attrs ~children in
  let get_properties m =
    let tag = @"attvalues" in
    let attrs = [] in
    let children =
      UTF8.Map.fold
        (fun n v acc ->
          let tag = @"attvalue" in
          let attrs = [ @"for", n; @"value", v ] in
          let children = [] in
          (XML.tag ~tag ~attrs ~children) :: acc)
        m
        [] in
    XML.tag ~tag ~attrs ~children in
  let edges, additional_vertices = simplify_edges g.edges g.vertices in
  let vertices = g.vertices @ additional_vertices in

  let meta =
    let tag = @"meta" in
    let attrs = [ @"lastmodifieddate", timestamp ] in
    let children = [ XML.tag
                       ~tag:@"creator"
                       ~attrs:[]
                       ~children:[XML.pcdata @"Barista"] ;
                     XML.tag
                       ~tag:@"description"
                       ~attrs:[]
                       ~children:[XML.pcdata @"Relationships between Java classes"] ] in
    XML.tag ~tag ~attrs ~children in

  let clusters = ref UTF8.Set.empty in
  let cluster_nodes =
    List.map
      (fun v ->
        match v.vertex_cluster with
        | Some c when (UTF8.length c > 0) && not (UTF8.Set.mem c !clusters) ->
            clusters := UTF8.Set.add c !clusters;
            let tag = @"node" in
            let attrs = [ @"id", c; @"label", c ] in
            let children = [] in
            [ XML.tag ~tag ~attrs ~children ]
        | _ -> [])
      vertices in
  let cluster_nodes = List.flatten cluster_nodes in
  let regular_nodes =
    List.map
      (fun v ->
        let tag = @"node" in
        let attrs =
          [ @"id", v.vertex_id ]
          @ (match v.vertex_label with
          | Some lbl -> [ @"label", lbl ]
          | None -> [])
          @ (match v.vertex_cluster with
          | Some c when UTF8.length c > 0 -> [ @"pid", c ]
          | _ -> []) in
        let children = [ get_properties v.vertex_properties ] in
        XML.tag ~tag ~attrs ~children)
      vertices in

  let all_edges =
    List.map
      (fun e ->
        let tag = @"edge" in
        let attrs =
          [ @"id", e.edge_id ]
          @ (match e.edge_vertices with
          | x :: y :: [] -> [ @"source", (fst x); @"target", (fst y) ]
          | _ -> [])
          @ [ @"type", (if e.edge_directed then @"directed" else @"undirected") ]
          @ (match e.edge_label with
          | Some lbl -> [ @"label", lbl ]
          | None -> []) in
        let children = [ get_properties e.edge_properties ] in
        XML.tag ~tag ~attrs ~children)
      edges in

  let graph =
    let tag = @"graph" in
    let attrs = [ @"defaultedgetype", @"directed" ] in
    let children = [
      get_attributes @"node" g.vertice_types ;
      get_attributes @"edge" g.edge_types ;
      XML.tag ~tag:@"nodes"
              ~attrs:[]
              ~children:(cluster_nodes @ regular_nodes) ;
      XML.tag ~tag:@"edges"
              ~attrs:[]
              ~children:all_edges
    ] in
    XML.tag ~tag ~attrs ~children in

  let document =
    let tag = @"gexf" in
    let attrs = [ @"xmlns", @"http://www.gexf.net/1.2draft" ;
                  @"xmlns:xsi", @"http://www.w3.org/2001/XMLSchema-instance" ;
                  @"xsi:schemaLocation", @"http://www.gexf.net/1.2draft http://www.gexf.net/1.2draft/gexf.xsd" ;
                  @"version", @"1.2" ] in
    let children = [ meta; graph ] in
    XML.tag ~tag ~attrs ~children in
  XML.dump document buff

let dump fmt g buff =
  match fmt with
  | Dot -> dump_dot g buff
  | GraphML -> dump_graphml g buff
  | GEXF -> dump_gexf g buff
