(*
 * This file is part of Barista.
 * Copyright (C) 2007-2014 Xavier Clerc.
 *
 * Barista is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Barista is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *)

(* Constraints imposed by {i LDC} instructions, that is maximal index of
   elements in the constant pool. *)


(** {6 Types} *)

type t =
  [ `Int of int32
  | `Float of float
  | `String of UTF8.t
  | `Class_or_interface of Name.for_class
  | `Array_type of Descriptor.array_type
  | `Method_type of Descriptor.for_method
  | `Method_handle of Bootstrap.method_handle ]
(** The type of constant pool elements that can impose a constraint. *)

type set
(** The type of constraint set. *)


(** {6 Exception} *)

BARISTA_ERROR =
  | Too_large of int
  | Too_many_constraints of int


(** {6 Functions} *)

val make : unit -> set
(** Constructs an new empty set of constraints. *)

val add : t -> set -> unit
(** [add c s] adds the constraint c to the set [s]. *)

val encode : set -> ConstantPool.extendable
(** [encode s] transforms the constraint set [s] into a constant pool that can
    be later used to encode methods imposing the constraints in [s].
    Raises [Exception] if the constant pool would be too large, or if there is
    too many constraints. *)

val equal : t -> t -> bool
(** Equality over constraints. *)

val compare : t -> t -> int
(** Comparison over constraints. *)

val hash : t -> int
(** Hash function over constraints. *)
