(*
 * This file is part of Barista.
 * Copyright (C) 2007-2014 Xavier Clerc.
 *
 * Barista is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Barista is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *)

(** Utility functions for code manipulation, and optimization. *)


val remove_dead_code : ('a, 'b) ControlFlow.graph -> ('a, 'b) ControlFlow.graph
(** [remove_dead_code g] returns [g] without the vertices that cannot be
    reached from the root. *)

val optimize_jumps : ('a, 'b) ControlFlow.graph -> ('a, 'b) ControlFlow.graph
(** [optimize_jumps g] returns [g] except that empty nodes ({i i. e.}
    with no instruction) have been short-circuited (but not removed,
    possibly producing dead code). *)

val optimize_switches : ('a, 'b) ControlFlow.graph -> ('a, 'b) ControlFlow.graph
(** [optimize_switches g] returns [g] except that lookup switches are
    replaced with table switches if it shorter. *)

val optimize_partial_evaluation : PartialEvaluation.t -> (int32 * Utils.u2 list, unit) ControlFlow.graph -> (int32 * Utils.u2 list, unit) ControlFlow.graph
(** [optimize_partial_evaluation pe g] optimizes [g] by applying partial
    evaluation techniques, using [pe] as the state at method begin. *)

type flattening_infos
(** The type used to label nodes of a flattened graph. *)

val offset_of_flattening_infos : flattening_infos -> int32
(** Returns the offset associated with the label of the passed flattened vertex. *)

val flattened_graph : (int32 * 'a, 'b) ControlFlow.graph -> (flattening_infos * 'a, 'b) ControlFlow.graph
(** Converts the passed graph, by assuming that it is already flattened,
    and that [int32] labels are vertices offsets. *)

val flatten_graph : ?use_offsets : bool -> (int32 * (Utils.u2 list), 'b) ControlFlow.graph -> Instruction.t list * (Utils.u2 * Utils.u2) list * Attribute.exception_table_element list * (flattening_infos * int32, 'b) ControlFlow.graph
(** [flatten_graph g] returns a [l, t, e, g'] quadruple where [g'] is [g]
    labeled with code offsets corresponding to the instruction list [l].
    [t] is the data for the {i LineNumberTable} attribute, and [e] is the
    exception table for [l]. *)

val optimize_graph : ?rules : Peephole.rewriting_rules list -> (('a * (Utils.u2 list)), 'b) ControlFlow.graph -> (('a * (Utils.u2 list)), 'b) ControlFlow.graph
(** [optimize_graph g] returns an optimized version of graph [g].
    It is equivalent to [remove_dead_code (optimize_switches (optimize_jumps (Peephole.optimize_graph g)))]. *)

val compute_stack_infos : Name.for_class -> StackState.instance StackState.unifier -> (flattening_infos * 'a, 'b) ControlFlow.graph -> StackState.t -> Utils.u2 * Utils.u2 * Attribute.stack_map_frame list
(** [compute_stack_infos cn u g s] returns [max_stack, max_locals, stack_map_frame]
    for the method whose control flow graph [g],initial stack state [s],
    and enclosing class [cn] are passed. The function [u] is used to
    unify stack states.

    Raises [StackState.Exception] if the passed instruction lists
    associated with the graph vertices are incoherent. *)
